/*
 * QTPixmap GTK engine
 *
 * This is a hacked/modified version of the Pixmap GTK engine. Modified to use
 * ~/.qt/qtrc to get colour information to recolour widgets.
 *
 * Changes are (C) Craig Drummond 2002 - Craig.Drummond@lycos.co.uk
 *
 */

#include "common.h"
#include <gmodule.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef GTK2_VER
#include <X11/Xlib.h>
#endif

#ifdef GTK2_VER
#define qt_theme_parse_color theme_parse_color
#define qt_theme_parse_int theme_parse_int
#define qt_theme_parse_bool theme_parse_bool
#define qt_theme_parse_size theme_parse_size
#define qt_theme_parse_stretch theme_parse_stretch
#define qt_theme_parse_rect theme_parse_rect
#endif

static int strcmp_i(const char *s1, const char *s2)
{
    char c1,
         c2;

    for(;;)
    {
        c1=*s1++;
        c2=*s2++;
        if(!c1 || !c2)
            break;
        if(isupper(c1))
            c1=tolower (c1);
        if(isupper(c2))
            c2=tolower (c2);
        if(c1!=c2)
            break;
    }
    return (int)c2-(int)c1;
}

struct qtpixmap_data qtpixmap_engine_data;

static int ctoh(char ch)
{
    return (ch>='0' && ch<='9') ? ch-'0' :
           (ch>='a' && ch<='f') ? 10+(ch-'a') :
           (ch>='A' && ch<='F') ? 10+(ch-'A') :
           0;
}

#define ATOH(str) ((ctoh(*str)<<4)+ctoh(*(str+1)))

static void set_rgb(struct qtpixmap_rgb *rgb, char *str)
{
    if(str && strlen(str)>5)
    {
        rgb->r=ATOH(str);
        rgb->g=ATOH(&str[2]);
        rgb->b=ATOH(&str[4]);
    }
    else
        rgb->r=rgb->g=rgb->b=0;
}

#define MAX_INPUT_LINE_LEN 1024
#define MAX_FILENAME_LEN   256

enum
{
    SECT_NONE,
    SECT_PALETTE,
    SECT_GENERAL,
    SECT_KWIN
};

/*
  Qt uses the following predefined weights: 
    Light    = 25,
    Normal   = 50,
    DemiBold = 63,
    Bold     = 75,
    Black    = 87

  ...need to categorize mid way, i.e. 0->37 = light, 38->56 = normal, etc...
*/

enum
{
    WEIGHT_NORMAL   = 38,   /* Less than this = light */
    WEIGHT_DEMIBOLD = 57,
    WEIGHT_BOLD     = 69,
    WEIGHT_BLACK    = 81
};

static const char * weight_str(int w)
{
    if(w<WEIGHT_NORMAL)
        return "light";
    else if(w<WEIGHT_DEMIBOLD)
#ifdef GTK2_VER
        return "";
#else
        return "medium";
#endif
    else if(w<WEIGHT_BOLD)
        return "demibold";
    else if(w<WEIGHT_BLACK)
        return "bold";
    else
        return "black";
}

#ifdef GTK2_VER
static const char * italic_str(int i)
{
    return i ? "Italic" : "";
}
#endif

enum
{
    RD_PALETTE     = 0x1,
    RD_FONT        = 0x2,
    RD_KWINPALETTE = 0x4
};

#ifndef GTK2_VER
static const char * iso8859_1_langs[]=
{
    "af", "afrikaans", "ca", "catalan", "da", "danish", "dansk", "de", "german", "deutsch",
    "en", "english", "es", "spanish", "eu", "basque", "fi", "finnish", "fo", "faroese",
    "faeroese", "fr", "french", "ga", "irish", "gd", "scottish", "gl", "galician", "is",
    "icelandic", "it", "italian", "nl", "dutch", "no", "norwegian", "pt", "portuguese",
    "sv", "swedish",
    NULL
};

static const char * iso8859_1_codesets[]=
{
    "ISO8859-1", "ISO_8859-1", "iso88591", "88591", "88591.en", "8859", "8859.in", "ascii",
    NULL
};

static const char * iso8859_2_langs[]=
{
    "cs", "croatian", "cz", "czech", "hr", "hu", "hungarian", "pl", "polish", "ro", "romanian",
    "rumanian", "serbocroatian", "sh", "sk", "sl", "slovak", "sr", "slovene", "slovenian",
    "sq", "albanian",
    NULL
};

static const char * iso8859_2_codesets[]=
{
    "ISO8859-2", "ISO_8859-2", "iso88592", "88592",
    NULL
};

static const char * iso8859_3_langs[] =
{
    "eo", "esperanto", "mt", "maltese",
    NULL
};

static const char * iso8859_3_codesets[]=
{
    "ISO8859-3", "ISO_8859-3", "iso88593", "88593",
    NULL
};

static const char * iso8859_5_langs[] =
{
    "be", "byelorussian", "bg", "bulgarian", "mk", "macedonian", "sp", 
    "sr", "serbian",
    NULL
};

static const char * iso8859_5_codesets[]=
{
    "ISO8859-5", "ISO_8859-5", "iso88595", "88595",
    NULL
};

static const char * iso8859_6_langs[] =
{
    "ar", "arabic",
    NULL
};

static const char * iso8859_6_codesets[]=
{
    "ISO8859-6", "ISO_8859-6", "iso88596", "88596",
    NULL
};

static const char * iso8859_7_langs[] =
{
    "el", "greek",
    NULL
};

static const char * iso8859_7_codesets[]=
{
    "ISO8859-7", "ISO_8859-7", "iso88597", "88597",
    NULL
};

static const char * iso8859_8_langs[] =
{
    "hebrew", "iw", "he",
    NULL
};

static const char * iso8859_8_codesets[]=
{
    "ISO8859-8", "ISO_8859-8", "iso88598", "88598",
    NULL
};

static const char * iso8859_9_langs[] =
{
    "tr", "tr", "turkish",
    NULL
};

static const char * iso8859_9_codesets[]=
{
    "ISO8859-9", "ISO_8859-9", "iso88599", "88599",
    NULL
};

static const char * iso8859_10_langs[] =
{
    "et", "estonian", "lt", "lithuanian", "lv", "latvian",
    NULL
};

static const char * iso8859_10_codesets[]=
{
    "ISO8859-10", "ISO_8859-10", "iso885910", "885910",
    NULL
};

static const char * iso8859_15_langs[] =
{
/*
      "fr", "french", "fi", "finnish",
*/
    NULL
};

static const char * iso8859_15_codesets[]=
{
    "ISO8859-15", "ISO_8859-15", "iso885915", "885915",
    NULL
};

static const char * koi8_r_langs[] =
{
    "ru", "russian",
    NULL
};

static const char * koi8_r_codesets[]=
{
    "KOI8-R", "KOI8_R",
    NULL
};

static const char * koi8_ru_langs[] =
{
    "uk", "ukrainian",
    NULL
};

static const char * koi8_ru_codesets[]=
{
    "KOI8-RU", "KOI8_RU",
    NULL
};

static struct
{
    const char *encoding;
    const char **languages,
               **codesets;
} x11EncodingMap[] =
    {
        { "iso8859-1",  iso8859_1_langs,  iso8859_1_codesets  },
        { "iso8859-2",  iso8859_2_langs,  iso8859_2_codesets  },
        { "iso8859-3",  iso8859_3_langs,  iso8859_3_codesets  },
        { "iso8859-5",  iso8859_5_langs,  iso8859_5_codesets  },
        { "iso8859-6",  iso8859_6_langs,  iso8859_6_codesets  },
        { "iso8859-7",  iso8859_7_langs,  iso8859_7_codesets  },
        { "iso8859-8",  iso8859_8_langs,  iso8859_8_codesets  },
        { "iso8859-9",  iso8859_9_langs,  iso8859_9_codesets  },
        { "iso8859-10", iso8859_10_langs, iso8859_10_codesets },
        { "iso8859-15", iso8859_15_langs, iso8859_15_codesets },
        { "KOI8-R",     koi8_r_langs,     koi8_r_codesets     },
        { "KOI8-RU",    koi8_ru_langs,    koi8_ru_codesets    },
        { NULL,         NULL,             NULL                }
    };

static const char * getX11Encoding()
{
    const char *enc=getenv("QT_GTK_ENC");

    if(!enc)
        enc=getenv("QTPIXMAP_GTK_ENC");

    if(!enc)
    {
        const char *locale;

        locale = getenv("LC_ALL");
        if (!locale || !*locale)
        { 
            locale = getenv("LC_CTYPE");
            if (!locale || !*locale)
                locale = getenv("LANG");
        }

        if (locale && *locale)
        {
            /* The most general syntax of a locale (not all optional parts recognized by all systems) is
               language[_territory][.codeset][@modifier][+special][,[sponsor][_revision]]
               To retrieve the codeset, search the first dot. Stop searching when
               a '@' or '+' or ',' is encountered. */
            char       *buf = (char*) malloc(strlen(locale)+1);
            const char *codeset = NULL;
            const char *cp = locale;

            for (; *cp != '\0' && *cp != '@' && *cp != '+' && *cp != ','; cp++)
            {
                if (*cp == '.')
                {
                    codeset = ++cp;

                    for (; *cp != '\0' && *cp != '@' && *cp != '+' && *cp != ','; cp++)
                        ;
                    if (*cp != '\0')
                    {
                        size_t n = cp - codeset;
                        memcpy(buf, codeset, n);
                        buf[n] = '\0';
                        codeset = buf;
                    }
                    break;
                }
            }

            if (codeset)
            {
                int i;

                for(i=0; x11EncodingMap[i].encoding && !enc; ++i)
                {
                    int j;

                    for(j=0; x11EncodingMap[i].codesets[j] && !enc; ++j)
                        if(0==strcmp_i(codeset, x11EncodingMap[i].codesets[j]))
                            enc=x11EncodingMap[i].encoding;
                }

                if(!enc)  /* Then didn't recognise codeset... */
                {
                    char *dash=strchr(codeset, '-');

                    /* Check if codeset is of the form <wibble>-<wobble> */
                    if(dash && strlen(dash)>1 && NULL==strchr(dash+1, '-'))  /* if so, then assume this is an OK X11 enc... */
                        enc=codeset;
                }
            }

            if(!enc) /* No matching/recognised codeset, so try to guess based upon language */
            {
                const char *underscore = strchr(locale, '_');
                const char *lang;
                int        i;

                if (underscore)
                {
                    size_t n = underscore - locale;
                    memcpy(buf,locale,n);
                    buf[n] = '\0';
                    lang = buf;
                }
                else
                    lang = locale;

                for(i=0; x11EncodingMap[i].encoding && !enc; ++i)
                {
                    int j;

                    for(j=0; x11EncodingMap[i].languages[j] && !enc; ++j)
                        if(0==strcmp_i(lang, x11EncodingMap[i].languages[j]))
                            enc=x11EncodingMap[i].encoding;
                }
            }

            free(buf);
        }
    }

    return enc ? enc : "iso8859-1";
}

static void getX11Res(int *resX, int *resY)
{
    gint width=gdk_screen_width(),
         height=gdk_screen_height(),
         widthMM=gdk_screen_width_mm(),
         heightMM=gdk_screen_height_mm();

    *resX=(width*254+widthMM*5)/(widthMM*10);
    *resY=(height*254+heightMM*5)/(heightMM*10);
}
#endif

int read_font_replacement(const char *file, const char **font)
{
    FILE *r=fopen(file, "r");
    int  ret=0;


    if(r)
    {
        char        line[MAX_INPUT_LINE_LEN+1];
        static char font_line[MAX_INPUT_LINE_LEN+1];

        while(NULL!=fgets(line, MAX_INPUT_LINE_LEN, r))
            if(line[0] != '#')
            {
                char *of=NULL;

                memcpy(font_line, line, MAX_INPUT_LINE_LEN+1);
                of=strtok(font_line, "=");
                if(0==strcmp_i(*font, of))
                {
                    *font=strtok(NULL, "\n");
                    ret=1;
                    break;
                }
            }
        fclose(r);
    }
    return ret;
}

#define SET_COLOR(COL, R, G, B) \
qtpixmap_engine_data.colors[COL].r=R; \
qtpixmap_engine_data.colors[COL].g=G; \
qtpixmap_engine_data.colors[COL].b=B;

static int read_rc(const char *rc, int rd)
{
    char       *home=getenv("HOME");
    int        found=0,
               weight=WEIGHT_NORMAL,
               italic=0,
               fixedW=0;
    float      size=12.0;
    const char *family="helvetica",
               *foundry="adobe";
    char       line[MAX_INPUT_LINE_LEN+1],
               font_line[MAX_INPUT_LINE_LEN+1];

    if(rd&RD_KWINPALETTE)
    {
        SET_COLOR(COLOR_KWIN_ABGND, 62, 145, 235)
        SET_COLOR(COLOR_KWIN_ABLEND, 62, 145, 235)
        SET_COLOR(COLOR_KWIN_AFGND, 255, 255, 255)
        SET_COLOR(COLOR_KWIN_ATBTN, 238, 238, 230)
        SET_COLOR(COLOR_KWIN_IBGND, 143, 159, 180)
        SET_COLOR(COLOR_KWIN_IBLEND, 143, 159, 180)
        SET_COLOR(COLOR_KWIN_IFGND, 255, 255, 255)
        SET_COLOR(COLOR_KWIN_ITBTN, 238, 238, 230)
        SET_COLOR(COLOR_KWIN_FRAME, 212, 208, 200)
        SET_COLOR(COLOR_KWIN_IFRAME, 212, 208, 200)
    }

    if(NULL!=home)
    {
        char fname[MAX_FILENAME_LEN+1];
        FILE *f;

        snprintf(fname, MAX_FILENAME_LEN, "%s/%s", home, rc);

        f=fopen(fname, "r");

        if(f)
        {
            int section=SECT_NONE;

            while(found!=rd && NULL!=fgets(line, MAX_INPUT_LINE_LEN, f))
                if(line[0]=='[')
                {
                    if(SECT_KWIN==section)
                        found|=RD_KWINPALETTE;

                    if(0==memcmp(line, "[Palette]", 9))
                        section=SECT_PALETTE;
                    else if(0==memcmp(line, "[General]", 9))
                        section=SECT_GENERAL;
                    else if(0==memcmp(line, "[KWinPalette]", 13))
                        section=SECT_KWIN;
                    else
                        section=SECT_NONE;
                }
                else if(SECT_PALETTE==section && rd&RD_PALETTE && !(found&RD_PALETTE) && 0==memcmp(line, "active=", 7))
                {
                    int  n=-1;
                    char *l=strtok(line, "#");

                    found|=RD_PALETTE;

                    while(l)
                    {
                        if(8==strlen(l))
                            switch(n)
                            {
                                case 0:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_FOREGROUND], l);
                                    break;
                                case 1:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_BUTTON], l);
                                    break;
                                case 5:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_MID], l);
                                    break;
                                case 6:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_TEXT], l);
                                    break;
                                case 9:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_BACKGROUND], l);
                                    break;
                                case 10:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_WINDOW], l);
                                    break;
                                case 12:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_SELECTED], l);
                                    break;
                                case 13:
                                    set_rgb(&qtpixmap_engine_data.colors[COLOR_TEXT_SELECTED], l);
                                    break;
                                default:
                                    break;
                            }
                        else
                            if(n>-1)
                                break;

                        n++;
                        if(n>13)
                            break;
                        l=strtok(NULL, "#");
                    }
                }
                else if (SECT_GENERAL==section && rd&RD_FONT && !(found&RD_FONT) && 0==memcmp(line, "font=", 5))
                {
                    int   n=-1,
                          rc_weight=WEIGHT_NORMAL,
                          rc_italic=0,
                          rc_fixedW=0;
                    float rc_size=12.0;
                    char  *l,
                          *rc_family=NULL,
                          *rc_foundry=NULL;

                    memcpy(font_line, line, MAX_INPUT_LINE_LEN+1);
                    l=strtok(font_line, "=");
                    found|=RD_FONT;

                    while(l)
                    {
                        switch(n)
                        {
                            case 0:  /* Family - and foundry(maybe!) (ignore X11 and XFT) */
                            {
                                char *ob=NULL,
                                     *cb=NULL;

                                if(NULL!=(ob=strchr(l, '[')) && ob!=l && NULL!=(cb=strchr(l, ']')))
                                {
                                    ob[-1]='\0';
                                    *cb='\0';
                                    rc_foundry=&(ob[1]);

                                    if(0==strcmp_i(rc_foundry, "xft") || 0==strcmp_i(rc_foundry, "x11"))
                                        rc_foundry=NULL;
                                }
                                else  /* Sometimes .kderc has "adobe-helvetica" */
                                {
                                    char *dash=NULL;

                                    if(NULL!=(dash=strchr(l, '-')))
                                    {
                                        rc_foundry=l;
                                        *dash='\0';
                                        l=++dash;
                                    }
                                }

                                rc_family=l;
                                break;
                            }
                            case 1:  /* Point size */
                                sscanf(l, "%f", &rc_size);
                                break;
                            case 4:  /* Weight */
                                sscanf(l, "%d", &rc_weight);
                                break;
                            case 5:  /* Slant */
                                sscanf(l, "%d", &rc_italic);
                                break;
                            case 8:  /* Spacing */
                                sscanf(l, "%d", &rc_fixedW);
                                break;
                            default:
                                break;
                        }

                        n++;
                        if(n>8 && NULL!=family)
                        {
                            weight=rc_weight;
                            italic=rc_italic;
                            fixedW=rc_fixedW;
                            size=rc_size;
                            family=rc_family;
                            foundry=rc_foundry;
                            break;
                        }
                        l=strtok(NULL, ",");
                    }
                }
                else if(SECT_KWIN==section && rd&RD_KWINPALETTE && !(found&RD_KWINPALETTE))
                {
                    char *num=strchr(line, '#');

                    if(num && ++num)
                        if(0==memcmp(line,      "activeBackground=", 17))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_ABGND], num);
                        else if(0==memcmp(line, "activeBlend=", 12))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_ABLEND], num);
                        else if(0==memcmp(line, "activeForeground=", 17))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_AFGND], num);
                        else if(0==memcmp(line, "activeTitleBtnBg=", 17))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_ATBTN], num);
                        else if(0==memcmp(line, "frame=", 6))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_FRAME], num);
                        else if(0==memcmp(line, "inactiveBackground=", 19))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_IBGND], num);
                        else if(0==memcmp(line, "inactiveBlend=", 14))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_IBLEND], num);
                        else if(0==memcmp(line, "inactiveForeground=", 19))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_IFGND], num);
                        else if(0==memcmp(line, "inactiveFrame=", 13))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_IFRAME], num);
                        else if(0==memcmp(line, "inactiveTitleBtnBg=", 19))
                            set_rgb(&qtpixmap_engine_data.colors[COLOR_KWIN_ITBTN], num);
                }
                else if(found==rd)
                    break;

            fclose(f);
        }
    }

    if(rd&RD_PALETTE && !(found&RD_PALETTE))
    {
        SET_COLOR(COLOR_BACKGROUND, 255, 255, 255)
        SET_COLOR(COLOR_BUTTON, 238, 234, 222)
        SET_COLOR(COLOR_SELECTED, 255, 221, 118)
        SET_COLOR(COLOR_WINDOW, 238, 238, 230)
        SET_COLOR(COLOR_FOREGROUND, 0, 0, 0)
        SET_COLOR(COLOR_MID, 198, 198, 191)
        SET_COLOR(COLOR_TEXT, 0, 0, 0)
        SET_COLOR(COLOR_TEXT_SELECTED, 0, 0, 0)
    }

    if(rd&RD_FONT)  /* No need to check if read in */
    {
#ifdef GTK2_VER
        qtpixmap_engine_data.font=(char *)malloc(
            strlen(family) + 1 +
            strlen(weight_str(weight)) + 1 +
            strlen(italic_str(italic)) + 1 +
            20+  /* point size */ +1);

        sprintf(qtpixmap_engine_data.font, "%s %s %s %d",
                family,
                weight_str(weight),
                italic_str(italic),
                (int)size);
#else
        char       *format="-%s-%s-%s-%s-*--*-%d-%d-%d-%s-*-%s";
        const char *encoding=getX11Encoding();
        int        resX,
                   resY;
        const char *global_name="/etc/X11/qt_gtk_fnt2fntrc";

        getX11Res(&resX, &resY);

        if(NULL!=home)
        {
            char home_name[MAX_FILENAME_LEN+1];

            snprintf(home_name, MAX_FILENAME_LEN, "%s/%s", home, "/.qt/gtk_fnt2fntrc");

            if(!read_font_replacement(home_name, &family))
                read_font_replacement(global_name, &family);
        }
        else
            read_font_replacement(global_name, &family);

        qtpixmap_engine_data.font=(char *)malloc(
            (foundry ? strlen(foundry) : 1) +
            strlen(family) +
            strlen(weight_str(weight))+
            1+   /* Italic */
            20+  /* point size */
            1+   /* fixed width */
            20+  /* Res X */
            20+  /* Res Y */
            strlen(format) +
            strlen(encoding));

        sprintf(qtpixmap_engine_data.font, format,
                foundry ? foundry : "*",
                family,
                weight_str(weight),
                italic ? "i" : "r",
                (int)(size*10),
                resX,
                resY,
                fixedW ? "m" : "p",
                encoding);
#endif
#if 0
printf("REQUEST FONT: %s\n", qtpixmap_engine_data.font);
#endif
    }

    return found;
}

static guint
theme_parse_color(GScanner * scanner,
                 enum fcolor *col)
{
  guint               token;

  token = g_scanner_get_next_token(scanner);
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_STRING)
    return G_TOKEN_STRING;

  if(0==strcmp_i(scanner->value.v_string, "background"))
      *col=COLOR_BACKGROUND;
  else if(0==strcmp_i(scanner->value.v_string, "button"))
      *col=COLOR_BUTTON;
  else if(0==strcmp_i(scanner->value.v_string, "selected"))
      *col=COLOR_SELECTED;
  else if(0==strcmp_i(scanner->value.v_string, "window"))
      *col=COLOR_WINDOW;
  else if(0==strcmp_i(scanner->value.v_string, "text"))
      *col=COLOR_TEXT;
  else if(0==strcmp_i(scanner->value.v_string, "foreground"))
      *col=COLOR_FOREGROUND;
  else if(0==strcmp_i(scanner->value.v_string, "text_selected"))
      *col=COLOR_TEXT_SELECTED;
  else if(0==strcmp_i(scanner->value.v_string, "mid"))
      *col=COLOR_MID;
  else if(0==memcmp(scanner->value.v_string, "kwin-", 5))
  {
      if(0==strcmp_i(scanner->value.v_string, "kwin-activeBackground"))
          *col=COLOR_KWIN_ABGND;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-activeBlend"))
          *col=COLOR_KWIN_ABLEND;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-activeForeground"))
          *col=COLOR_KWIN_AFGND;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-activeTitleBtnBg"))
          *col=COLOR_KWIN_ATBTN;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-frame"))
          *col=COLOR_KWIN_FRAME;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-inactiveBackground"))
          *col=COLOR_KWIN_IBGND;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-inactiveBlend"))
          *col=COLOR_KWIN_IBLEND;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-inactiveForeground"))
          *col=COLOR_KWIN_IFGND;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-inactiveFrame"))
          *col=COLOR_KWIN_IFRAME;
      else if(0==strcmp_i(scanner->value.v_string, "kwin-inactiveTitleBtnBg"))
          *col=COLOR_KWIN_ITBTN;
      else
          *col=COLOR_NONE;
  }
  else
      *col=COLOR_NONE;

  return G_TOKEN_NONE;
}

static guint
theme_parse_int(GScanner * scanner,
                int *i)
{
  guint token;

  token = g_scanner_get_next_token(scanner);
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);

  if (token != G_TOKEN_INT && token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  *i = G_TOKEN_STRING==token ? atoi(scanner->value.v_string) : scanner->value.v_int;

  return G_TOKEN_NONE;
}

static guint
theme_parse_bool(GScanner * scanner,
                gboolean *b)
{
  guint token;

  token = g_scanner_get_next_token(scanner);
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);

  if (token == TOKEN_TRUE)
    *b = TRUE;
  else if (token == TOKEN_FALSE)
    *b = FALSE;
  else
    return TOKEN_TRUE;

  return G_TOKEN_NONE;
}

static guint
theme_parse_size(GScanner * scanner,
                 struct image_size *size)
{
  guint token;

  token = g_scanner_get_next_token(scanner);
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_LEFT_CURLY)
    return G_TOKEN_LEFT_CURLY;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_INT)
    return G_TOKEN_INT;
  size->width = scanner->value.v_int;
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_COMMA)
    return G_TOKEN_COMMA;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_INT)
    return G_TOKEN_INT;
  size->height = scanner->value.v_int;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_RIGHT_CURLY)
    return G_TOKEN_RIGHT_CURLY;

  return G_TOKEN_NONE;
}

static guint
theme_parse_stretch(GScanner * scanner,
                enum fstretch *s)
{
  guint token;

  token = g_scanner_get_next_token(scanner);
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);

  if (token == TOKEN_TRUE)
    *s = STRETCH_BOTH;
  else if (token == TOKEN_FALSE)
    *s = STRETCH_NONE;
  else
  {
    if (token != G_TOKEN_STRING)
      return G_TOKEN_STRING;

    if(0==strcmp_i(scanner->value.v_string, "true") || 0==strcmp_i(scanner->value.v_string, "both"))
      *s = STRETCH_BOTH;
    else if(0==strcmp_i(scanner->value.v_string, "false") || 0==strcmp_i(scanner->value.v_string, "none"))
      *s = STRETCH_NONE;
    else if(0==strcmp_i(scanner->value.v_string, "width"))
      *s = STRETCH_WIDTH;
    else if(0==strcmp_i(scanner->value.v_string, "height"))
      *s = STRETCH_HEIGHT;
    else
      *s = STRETCH_NONE;
  }

  return G_TOKEN_NONE;
}

static guint
theme_parse_rect(GScanner * scanner,
                 struct qtp_rect *r)
{
  guint token;

  token = g_scanner_get_next_token(scanner);
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_LEFT_CURLY)
    return G_TOKEN_LEFT_CURLY;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_INT && token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  r->x = G_TOKEN_STRING==token ? atoi(scanner->value.v_string) : scanner->value.v_int;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_COMMA)
    return G_TOKEN_COMMA;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_INT && token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  r->y = G_TOKEN_STRING==token ? atoi(scanner->value.v_string) : scanner->value.v_int;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_COMMA)
    return G_TOKEN_COMMA;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_INT && token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  r->w = G_TOKEN_STRING==token ? atoi(scanner->value.v_string) : scanner->value.v_int;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_COMMA)
    return G_TOKEN_COMMA;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_INT && token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  r->h = G_TOKEN_STRING==token ? atoi(scanner->value.v_string) : scanner->value.v_int;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_RIGHT_CURLY)
    return G_TOKEN_RIGHT_CURLY;

  return G_TOKEN_NONE;
}

static int qtp_refs = 0;

static void qtp_init()
{
  if(0==qtp_refs)
  {
#ifndef GTK2_VER
    qtpixmap_engine_data.button_x_offset=qtpixmap_engine_data.button_y_offset=
    qtpixmap_engine_data.tb_button_x_offset=qtpixmap_engine_data.tb_button_y_offset=0;
    qtpixmap_engine_data.radio_check_focus_mod.x=
    qtpixmap_engine_data.radio_check_focus_mod.y=
    qtpixmap_engine_data.radio_check_focus_mod.w=
    qtpixmap_engine_data.radio_check_focus_mod.h=
#endif
    qtpixmap_engine_data.button_focus_mod.x=
    qtpixmap_engine_data.button_focus_mod.y=
    qtpixmap_engine_data.button_focus_mod.w=
    qtpixmap_engine_data.button_focus_mod.h=
    qtpixmap_engine_data.combo_focus_mod.x=
    qtpixmap_engine_data.combo_focus_mod.y=
    qtpixmap_engine_data.combo_focus_mod.w=
    qtpixmap_engine_data.combo_focus_mod.h=0;
    qtpixmap_engine_data.font=NULL;

    qtpixmap_engine_data.no_radio_check_highlight=TRUE;
    qtpixmap_engine_data.use_selected_for_menu_item=TRUE;
    qtpixmap_engine_data.use_selected_for_menu=FALSE;

    read_rc(".qt/qtrc", RD_PALETTE|RD_FONT|RD_KWINPALETTE);
  }

  qtp_refs++;
}

static void qtp_exit()
{
  qtp_refs--;
  if(0==qtp_refs)
  {
    if(qtpixmap_engine_data.font)
      free(qtpixmap_engine_data.font);
    qtpixmap_engine_data.font=NULL;
  }
}

static void set_colour(GdkColor *gtk_col, enum fcolor col)
{
    gtk_col->red=qtpixmap_engine_data.colors[col].r<<8;
    gtk_col->green=qtpixmap_engine_data.colors[col].g<<8;
    gtk_col->blue=qtpixmap_engine_data.colors[col].b<<8;
}

#define SET_COLOUR(item, ITEM, state, QTP_COL) \
    if(rc_style->color_flags[state]&ITEM) \
        style->item[state]=rc_style->item[state]; \
    else \
        set_colour(&(style->item[state]), QTP_COL);

static void set_colours(GtkStyle *style, GtkRcStyle *rc_style)
{
    SET_COLOUR(bg, GTK_RC_BG, GTK_STATE_NORMAL, COLOR_WINDOW)
    SET_COLOUR(bg, GTK_RC_BG, GTK_STATE_SELECTED, COLOR_SELECTED)
    SET_COLOUR(bg, GTK_RC_BG, GTK_STATE_INSENSITIVE, COLOR_WINDOW)
    SET_COLOUR(bg, GTK_RC_BG, GTK_STATE_ACTIVE, COLOR_MID)
    SET_COLOUR(bg, GTK_RC_BG, GTK_STATE_PRELIGHT, COLOR_WINDOW)

    SET_COLOUR(base, GTK_RC_BASE, GTK_STATE_NORMAL, COLOR_BACKGROUND)
    SET_COLOUR(base, GTK_RC_BASE, GTK_STATE_SELECTED, COLOR_SELECTED)
    SET_COLOUR(base, GTK_RC_BASE, GTK_STATE_INSENSITIVE, COLOR_WINDOW)
    SET_COLOUR(base, GTK_RC_BASE, GTK_STATE_ACTIVE, COLOR_SELECTED)
    SET_COLOUR(base, GTK_RC_BASE, GTK_STATE_PRELIGHT, COLOR_BACKGROUND)

    SET_COLOUR(text, GTK_RC_TEXT, GTK_STATE_NORMAL, COLOR_TEXT)
    SET_COLOUR(text, GTK_RC_TEXT, GTK_STATE_SELECTED, COLOR_TEXT_SELECTED)
    SET_COLOUR(text, GTK_RC_TEXT, GTK_STATE_INSENSITIVE, COLOR_MID)
    SET_COLOUR(text, GTK_RC_TEXT, GTK_STATE_ACTIVE, COLOR_TEXT_SELECTED)
    SET_COLOUR(text, GTK_RC_TEXT, GTK_STATE_PRELIGHT, COLOR_TEXT)

    SET_COLOUR(fg, GTK_RC_FG, GTK_STATE_NORMAL, COLOR_FOREGROUND)
    SET_COLOUR(fg, GTK_RC_FG, GTK_STATE_SELECTED, COLOR_TEXT_SELECTED)
    SET_COLOUR(fg, GTK_RC_FG, GTK_STATE_INSENSITIVE, COLOR_MID)
    SET_COLOUR(fg, GTK_RC_FG, GTK_STATE_ACTIVE, COLOR_FOREGROUND)
    SET_COLOUR(fg, GTK_RC_FG, GTK_STATE_PRELIGHT, COLOR_FOREGROUND)
}
