/* $Id: dnsutil.c,v 1.49 2004/11/23 19:45:34 graziano Exp $ */

#include "config_portability.h"

#include <stdio.h>
#include <stdlib.h>     /* REALLOC() */
#include <string.h>
#include <sys/types.h>  /* sometimes required for #include <sys/socket.h> */
#include <sys/socket.h> /* AF_INET */
#include <netinet/in.h> /* struct in_addr */
#include <arpa/inet.h>  /* inet_addr() inet_ntoa() */
#include <netdb.h>      /* {end,set}hostent() gethostby{addr,name}() */
#include <errno.h>

#include "diagnostic.h"
#include "osutil.h"
#include "dnsutil.h"
#include "protocol.h"
#include "strutil.h"


/*
** NOTE: The man pages for {end,set}hostent() seem to imply that endhostent()
** needs/should only be called after sethostent(1).  Below, we call
** endhostent() after calling sethostent(0).  So far, this hasn't seemed to
** cause any problems, and it also appears to have squished a bug on some
** version of Unix where the O/S DNS cache was losing entries.
*/

static void *lock = NULL;		/* local lock */

/*
 * We cache host entries locally to avoid going to the DNS too often.  This
 * also gets around an old Solaris bug which leaks memory whenever dlopen is
 * called (such as on the dynamic DNS lookup library). 
 *
 * Cached values timeout after CACHE_TIMEOUT seconds.
 *
 * We keep at most CACHE_MAX slots.
 */
#define CACHE_TIMEOUT 1800
#define CACHE_MAX 512
static unsigned int cacheCount = 0;
static struct hostent **cache = NULL;
static double *cacheTimeout = NULL;


/*
 * Looks in the name and alias entries of #hostEntry# for a fully-qualified
 * name.  Returns the fqn if found; otherwise, returns the name entry.
 */
static const char *
BestHostName(const struct hostent *hostEntry) {
	int i;

	if (hostEntry == NULL) {
		return NULL;
	}

	if (!strchr(hostEntry->h_name, '.')) {
		for (i = 0; hostEntry->h_aliases[i] != NULL; i++) {
			if (strchr(hostEntry->h_aliases[i], '.'))
				return hostEntry->h_aliases[i]; /* found! */
		}
	}

	/* If we don't have a fully-qualified name, do the best we can.  */
	return hostEntry->h_name;
}

/* 
 * free a struct hostent *. 
 */
static void
FreeStructHostent(struct hostent *toFree) {
	int i;

	/* sanity check */
	if (toFree == NULL) {
		return;
	}

	/* let's start to free */
	if (toFree->h_aliases != NULL) {
		for (i=0; toFree->h_aliases[i] != NULL; i++) {
			free(toFree->h_aliases[i]);
		}
		free(toFree->h_aliases);
	}
	if (toFree->h_name != NULL) {
		free(toFree->h_name);
	}
	if (toFree->h_addr_list != NULL) {
		for (i=0; toFree->h_addr_list[i] != NULL; i++) {
			free(toFree->h_addr_list[i]);
		}
		free(toFree->h_addr_list);
	}
	free(toFree);

	return;
}

/*
 * copy a struct hostent * into a newly allocated struct hostent * that
 * you need to free when you are done using. Returns NULL in case of
 * errors.
 */
static struct hostent *
CopyStructHostent(const struct hostent *orig) {
	struct hostent *ret;
	int i,j;

	/* sanity check */
	if (orig == NULL) {
		return NULL;
	}

	ret = (struct hostent *)MALLOC(sizeof(struct hostent));
	if (ret == NULL) {
		ERROR("Out of memory\n");
		return NULL;		/* out of memory */
	}
	memset((void *)ret, 0,  sizeof(struct hostent));
	
	/* make room for the name */
	ret->h_name = strdup(orig->h_name);
	if (ret->h_name == NULL) {
		free(ret);
		ERROR("Out of memory\n");
		return NULL; 		/* out of memory */
	}

	/* count aliases and copy them */
	for (i=0; orig->h_aliases != NULL && orig->h_aliases[i] != NULL; i++) {
		;
	}
	ret->h_aliases = (char **)MALLOC(sizeof(char *) * (i+1));
	if (ret->h_aliases == NULL) {
		FreeStructHostent(ret);
		ERROR("Out of memory\n");
		return NULL;
	}
	for (j=0; j < i; j++) {
		ret->h_aliases[j] = strdup(orig->h_aliases[j]);
		if (ret->h_aliases[j] == NULL) {
			FreeStructHostent(ret);
			ERROR("Out of memory\n");
			return NULL;
		}
	}
	ret->h_aliases[i] = NULL;

	/* copy the easy stuff */
	ret->h_addrtype = orig->h_addrtype;
	ret->h_length = orig->h_length;

	/* copy the addresses */
	for (i=0; orig->h_addr_list != NULL && orig->h_addr_list[i] != NULL; i++) {
		;
	}
	ret->h_addr_list = (char **)MALLOC(sizeof(struct in_addr *) * (i+1));
	if (ret->h_addr_list == NULL) {
		FreeStructHostent(ret);
		ERROR("Out of memory\n");
		return NULL;
	}
	for (j=0; j < i; j++) {
		ret->h_addr_list[j] = (char *)MALLOC(ret->h_length + 1);
		if (ret->h_addr_list[j] == NULL) {
			FreeStructHostent(ret);
			ERROR("Out of memory\n");
			return NULL;
		}
		memcpy(ret->h_addr_list[j], orig->h_addr_list[j], ret->h_length);
	}
	ret->h_addr_list[i] = NULL;

	/* done */
	return ret;
}


/*
 * Appends #hostEntry# (a copy) to the global map cache. 
 *
 * cache is a global structure and need to be protected by locks to be
 * thread safe.
 */
static void
CacheHostent(struct hostent *hostEntry) {
	int ind, toFree;
	struct hostent **extendedCache;
	double *tmp_touts, now;


	/* sanity check */
	if (hostEntry == NULL) {
		return;
	}
	
	now = CurrentTime();
	if (!GetNWSLock(&lock)) {
		ERROR("CacheHostent: couldn't obtain the lock\n");
	}

	/* let's see if we have room to add the entry */
	if (cacheCount >= CACHE_MAX) {
		/* look for an expired entry */
		for (ind = 0, toFree = -1; ind < cacheCount; ind++) {
			if (cache[ind] == NULL) {
				break;
			}
			if (now > cacheTimeout[ind]) {
				toFree = ind;
			}
		}

		/* let's see which one we can remove */
		if (ind >= cacheCount) {
			if (toFree == -1) {
				/* we need to get rid of one entry */
				toFree = ((int)now) % CACHE_MAX;
			}
			ind = toFree;
			FreeStructHostent(cache[ind]);
		}
	} else {
		/* no found, we need to add memory */
		ind = cacheCount;
		cacheCount++;
		extendedCache = (struct hostent**)REALLOC(cache, sizeof(struct hostent *) * cacheCount);
		tmp_touts = (double *)REALLOC(cacheTimeout, sizeof(double) * cacheCount);
		if(extendedCache == NULL || tmp_touts == NULL) {
			ReleaseNWSLock(&lock);
			ERROR("Out of memory\n");
			return;
		}
		cache = extendedCache;
		cacheTimeout = tmp_touts;
	}

	cache[ind] = CopyStructHostent(hostEntry);
	if (cache[ind] == NULL) {
		cacheTimeout[ind] = 1;
	} else {
		cacheTimeout[ind] = now + CACHE_TIMEOUT;
	}
	ReleaseNWSLock(&lock);

	return;
}


/*
 * Searches the DNS mapping cache for #address#, adding a new entry 
 * if needed.  Returns a copy of the the mapping entry, or 
 * NULL on error. The memory returned needs to be freed.
 */
static struct hostent*
LookupByAddress(IPAddress address) {
	struct in_addr addr;
	struct hostent *tmp, *addrEntry;
	struct in_addr **cachedAddr;
	int i;
	double now;

	/* look if we have it in the cache and it's not expired */
	now = CurrentTime();
	if (!GetNWSLock(&lock)) {
		ERROR("LookupByAddress: couldn't obtain the lock\n");
	}
	for (i = 0; i < cacheCount; i++) {
		if (cache[i] == NULL) {
			continue;
		}

		for (cachedAddr = (struct in_addr**)cache[i]->h_addr_list;
				*cachedAddr != NULL; cachedAddr++) {
			if ((**cachedAddr).s_addr == address.addr) {
				break;
			}
		}
		/* let's see if we found it */
		if (*cachedAddr != NULL) {
			break;
		}
	}

	/* if we found something and it's current we are done */
	if (i < cacheCount && now < cacheTimeout[i]) {
		addrEntry = CopyStructHostent(cache[i]);
		ReleaseNWSLock(&lock);
		return addrEntry;
	}

	/* we need to search */
	addr.s_addr = address.addr;
	sethostent(0);
	tmp = gethostbyaddr((char *)&addr, sizeof(addr), AF_INET);
	addrEntry = CopyStructHostent(tmp);
	endhostent();
	ReleaseNWSLock(&lock);

	if (addrEntry == NULL) {
		/* we are done */
		INFO1("LookupByAddress: couldn't resolve (%u)\n", address.addr);
	} else if (addrEntry->h_length != sizeof(struct in_addr)) {
		/* We don't (yet) handle non-in_addr addresses. */
		FreeStructHostent(addrEntry);
		addrEntry = NULL;
	} else {
		CacheHostent(addrEntry);
	}

	/* now if we have an expired entry, but not a new one, we use the
	 * old one at least to return something */
	if (addrEntry == NULL && i < cacheCount) {
		INFO1("LookupByAddress: using old entry for %s\n", cache[i]->h_name);
		addrEntry = CopyStructHostent(cache[i]);
	}

	/* addrEntry will need to be freed after */
	return addrEntry;
}


/*
 * Searches the DNS mapping cache for #name#, adding a new entry if needed.
 * Returns a pointer to the mapping entry, or NULL on error. The returned
 * value need to be freed.
 */
static struct hostent*
LookupByName(const char *needle) {
	char **name;
	struct hostent *tmp, *nameEntry;
	double now;
	int i;

	/* look in the cache for non expired entry */
	now = CurrentTime();
	if (!GetNWSLock(&lock)) {
		ERROR("LookupByName: failed to obtain the lock\n");
	}
	for(i = 0; i < cacheCount; i++) {
		if (cache[i] == NULL) {
			continue;
		}
		/* look in the canonic name */
 		if(strcasecmp(needle, cache[i]->h_name) == 0) {
			break;
		}
		/* and the aliases */
		for(name = cache[i]->h_aliases; *name != NULL; name++) {
			if(strcasecmp(*name, needle) == 0) {
				break;
			}
		}
		/* let's see if we found it */
		if(*name != NULL) {
			break;
		}
	}

	/* if we found something and it's current we are done */
	if (i < cacheCount && now < cacheTimeout[i]) {
		nameEntry = CopyStructHostent(cache[i]);
		ReleaseNWSLock(&lock);
		return nameEntry;
	}

	/* we need to query DNS */
	sethostent(0);
	tmp = gethostbyname(needle);
	nameEntry = CopyStructHostent(tmp);
	endhostent();

	ReleaseNWSLock(&lock);

	if(nameEntry == NULL) {
		INFO1("LookupByName: failed to resolve %s\n", needle);
	} else if(nameEntry->h_length != sizeof(struct in_addr)) {
		/* We don't (yet) handle non-in_addr addresses. */
		INFO("LookupByName: not sure what I got back!\n");
		FreeStructHostent(nameEntry);
		nameEntry = NULL;
	} else {
		/* I'm not comletely sure why we do this, but since it
		 * was there before me, I won't kill it. Let's check of
		 * the nickname (the name we searched) is already there,
		 * if not add it */

		/* let's count the aliases */
		for (i = 0; nameEntry->h_aliases[i] != NULL; i++) {
			;
		}

		/* let's see if we already have it */
		for(name = nameEntry->h_aliases; *name != NULL; name++) {
			if(strcasecmp(*name, needle) == 0) {
				break;
			}
		}
		/* if not add it */
		if (*name == NULL) {
			name = (char **)REALLOC(nameEntry->h_aliases, sizeof(char **) * (i + 2));
			if (name == NULL) {
				ERROR("Out of memory\n");
				return NULL;
			}
			name[i] = strdup(needle);
			if (name[i] == NULL) {
				FREE(name);
				ERROR("Out of memory\n");
				return NULL;
			}
			name[i + 1] = NULL;
			nameEntry->h_aliases = name;

			/* since we add an alias, let's remove a possibly
			 * old cache entry */
			GetNWSLock(&lock);
			for (i = 0; i < cacheCount; i++) {
				if (cache[i] == NULL) {
					continue;
				}
				if(strcmp(nameEntry->h_name, cache[i]->h_name) == 0) {
					break;
				}
			}
			if (i < cacheCount) {
				FreeStructHostent(cache[i]);
				cache[i] = NULL;
			}
			ReleaseNWSLock(&lock);
		}

		/* finally we push the new entry into the cache */
		CacheHostent(nameEntry);
	}

	/* now if we have an expired entry, but not a new one, we use the
	 * old one at least to return something */
	if (nameEntry == NULL && i < cacheCount) {
		INFO1("LookupByName: using old entry for %s\n", cache[i]->h_name);
		nameEntry = CopyStructHostent(cache[i]);
	}

	/* nameEntry will need to be freed */
	return nameEntry;
}

/* thread safe */
IPAddress
Peer(Socket sd) {
	struct sockaddr peer;
	SOCKLEN_T peer_size = sizeof(peer);
	IPAddress addr;

	addr.addr = 0;

	if (!IsPipe(sd) && (getpeername(sd, &peer, &peer_size) == 0)) {
		if (peer.sa_family == AF_INET) {
			addr.addr = (((struct sockaddr_in *)&peer)->sin_addr.s_addr);
		}
	}

	return addr;
}

/* thread safe */
char *
PeerName_r(Socket sd) {
	struct sockaddr peer;
	SOCKLEN_T peer_size = sizeof(peer);
	char *returnValue;
	IPAddress addr;

	if (IsPipe(sd)) {
		returnValue = strdup("pipe");
	} else if (getpeername(sd, &peer, &peer_size) < 0) {
		DDEBUG2("PeerName_r: gepeername error (%d %s)\n", errno, strerror(errno));
		returnValue = strdup("unknown");
	} else {
		if (peer.sa_family != AF_INET) {
			returnValue = strdup("unknown");
		} else {
			addr.addr = ((struct sockaddr_in *)&peer)->sin_addr.s_addr;
			returnValue = IPAddressImage_r(addr);
		}
	}

	if (returnValue == NULL) {
		ERROR("PeerName_r: out of memory\n");
	}
	return returnValue;
}


/* thread safe */
unsigned short
PeerNamePort(Socket sd) {
	unsigned short tmp = 0;
	struct sockaddr peer;
	SOCKLEN_T peer_size = sizeof(peer);

	/* connectedPipes is global */
	if (!GetNWSLock(&lock)) {
		ERROR("PeerNamePort: failed to obtain the lock\n");
	}
	if (!IsPipe(sd) && getpeername(sd, &peer, &peer_size) == 0) {
		tmp = ((struct sockaddr_in *)&peer)->sin_port;
	}
	ReleaseNWSLock(&lock);
	return tmp;
}


/* thread safe: we allocate memory for the returned char * */
char *
IPAddressImage_r(IPAddress addr) {
	struct in_addr addrAsInAddr;
	char *returned, *tmp;

	addrAsInAddr.s_addr = addr.addr;
	returned = NULL;

	if (!GetNWSLock(&lock)) {
		ERROR("IPAddressImage_r: failed to obtain the lock\n");
	}
	tmp = inet_ntoa(addrAsInAddr);
	if (tmp != NULL) {
		returned = strdup(tmp);
		if (returned == NULL) {
			ABORT("IPAddressImage_r: out of memory\n");
		}
	} else {
		ERROR1("IPAddressImage_r: failed to convert %d\n", addr.addr);
	}
	ReleaseNWSLock(&lock);

	return returned;
}


/* thread safe: we allocate memory for the returned char. NULL have to be
 * checked by the caller. */
char *
IPAddressMachine_r(IPAddress addr) {
	struct hostent *hostEntry;
	char *returnValue;

	hostEntry = LookupByAddress(addr);
	if (hostEntry == NULL) {
		return NULL;
	}

	returnValue = strdup(BestHostName(hostEntry));
	if (returnValue != NULL) {
		/* change to lower case */
		strcase(returnValue, ALL_LOWER);
	} else {
		WARN("IPAddressMachine_r: out of memory!\n");
	}

	/* free allocated memory */
	FreeStructHostent(hostEntry);

	return returnValue;
}


/* thread safe. */
int
IPAddressValues(const char *machineOrAddress,
                IPAddress *addressList,
                unsigned int atMost) {
	struct hostent *hostEntry;
	int i = 0, itsAnAddress = 0;
	IPAddress temp;
#ifdef HAVE_INET_ATON
	struct in_addr in;

	if (inet_aton(machineOrAddress, &in) != 0) {
		itsAnAddress = 1;
		temp.addr = in.s_addr;
	} else {
		/* invalid */
		temp.addr = -1;
	}
#else
	/* inet_addr() has the weird behavior of returning an unsigned
	 * quantity but using -1 as an error value.  Furthermore, the
	 * value returned is sometimes int and sometimes long,
	 * complicating the test.  Once inet_aton() is more widely
	 * available, we should switch to using it instead.  */
	temp.addr = inet_addr(machineOrAddress);
	if (temp.addr != -1) {
		itsAnAddress = 1;
	}
#endif

	/* let's search for name/address */
	if (itsAnAddress) {
		hostEntry = LookupByAddress(temp);
		if (hostEntry == NULL && temp.addr != -1) {
			/* we couldn't fetch the name: use what we have */
			if (atMost > 0) {
				*addressList = temp;
			}
			return 1;
		}
	} else {
		hostEntry = LookupByName(machineOrAddress);
	}

	/* sanity check */
	if(hostEntry == NULL) {
		return 0;
	} 

	/* if atMost == 0 means we are checking if the address is
	 * correct. It is */
	if(atMost == 0) {
		i = 1;
	} 

	for(; i < atMost && hostEntry->h_addr_list[i] != NULL; i++) {
		memcpy(&(addressList[i].addr), hostEntry->h_addr_list[i], hostEntry->h_length);
	}
	FreeStructHostent(hostEntry);

	return i;
}


/* well, the name is always the same so we can use a static variable.
 * Changed to never return NULL but at the very worse localhost. */
const char *
MyMachineName(void) {

	struct hostent *myEntry;
	static char returnValue[255] = "";

	/* If we have a value in returnValue, done */
	if(returnValue[0] != '\0') {
		return returnValue;
	}

	/* try the simple case first */
	if(gethostname(returnValue, sizeof(returnValue)) == -1) {
		ERROR("gethostname() failed! using localhost instead.\n");
		/* setting the name to a safe bet */
		if (!GetNWSLock(&lock)) {
			ERROR("MyMachineName: failed to obtain the lock\n");
		}
		strncpy(returnValue, "localhost", sizeof(returnValue));
		ReleaseNWSLock(&lock);
	} else if(!strchr(returnValue, '.')) {
		/* Okay, that didn't work; try the DNS. */
		myEntry = LookupByName(returnValue);
		if(myEntry != NULL) {
			if (!GetNWSLock(&lock)) {
				ERROR("MyMachineName: failed to obtain the lock\n");
			}
			strncpy(returnValue,BestHostName(myEntry),sizeof(returnValue));
			returnValue[sizeof(returnValue) - 1] = '\0';
			ReleaseNWSLock(&lock);
			FreeStructHostent(myEntry);
		}
	}

	/* let's get our name lowercase */
	strcase(returnValue, ALL_LOWER);

	return returnValue;
}

