/* $Id: nws_search.c,v 1.48 2004/09/30 20:42:46 graziano Exp $ */

#include "config_nws.h"
#include <stdio.h>       /* printf() */
#include <stdlib.h>
#include <string.h>      /* string functions */
#include <strings.h>     /* strncasecmp() on aix */
#include <unistd.h>      /* getopt() */

#define NWSAPI_SHORTNAMES
#include "host_protocol.h"
#include "register.h"
#include "nws_api.h"     /* NWS programming interface */
#include "diagnostic.h"  /* ABORT() */


/*
** This program allows command-line query of NWS registrations.  See the
** USERSGUIDE for a description of the command-line options.
*/


#define ACTIVITY_SHORTCUT "activities"
#define CLIQUE_SHORTCUT "cliques"
#define CONTROL_SHORTCUT "controls"
#define FORECASTER_SHORTCUT "forecasters"
#define HOST_SHORTCUT "hosts"
#define MEMORY_SHORTCUT "memories"
#define MEMORY_SHORTCUT2 "memory"
#define SENSOR_SHORTCUT "sensors"
#define SERIES_SHORTCUT "series"
#define SKILL_SHORTCUT "skills"


const char *FILTERS[] =
  {ACTIVITIES, CLIQUES, CONTROLS, FORECASTERS, HOSTS, MEMORIES,
   MEMORIES, SENSORS, SERIES, SKILLS};
const char *SHORTCUTS[] =
  {ACTIVITY_SHORTCUT, CLIQUE_SHORTCUT, CONTROL_SHORTCUT, FORECASTER_SHORTCUT,
   HOST_SHORTCUT, MEMORY_SHORTCUT, MEMORY_SHORTCUT2, SENSOR_SHORTCUT,
   SERIES_SHORTCUT, SKILL_SHORTCUT};


/*
** Returns the length of the longest attribute name in #obj#.
*/
int
LongestNameLength(const Object obj) {
	NwsAttribute attr;
	int nameLen;
	int returnValue = 0;
	char *tmp;

	ForEachNwsAttribute(obj, attr) {
		tmp = NwsAttributeName_r(attr);
		if (tmp == NULL) {
			continue;
		}
		nameLen = strlen(tmp);
		if(nameLen > returnValue)
			returnValue = nameLen;

		free(tmp);
	}

	return returnValue;
}


/*
** Prints to stdout the name and/or value of #attr#, depending on #whatToPrint#.
** If printing names, pads the name with blanks to #namePadding# characters.
*/
typedef enum {NAMES_AND_VALUES, NAMES_ONLY, VALUES_ONLY}
  NwsAttributeCharacteristics;
static void
PrintNwsAttribute(NwsAttribute attr,
               NwsAttributeCharacteristics whatToPrint,
               int namePadding) {

	const char *c;
	char *tmp;
	int i;
	int nameLen;

	/* get the name of the attribute */
	tmp = NwsAttributeName_r(attr);
	if(whatToPrint != VALUES_ONLY) {
		printf("%s", tmp);
	}

	/* if we need to print values too lets pad the name */
	if(whatToPrint == NAMES_AND_VALUES) {
		nameLen = strlen(tmp);
		for(i = nameLen; i < namePadding; i++)
			putchar(' ');
		putchar(':');
	}
	/* no more use for the name */
	free(tmp);

	if(whatToPrint != NAMES_ONLY) {
		tmp = NwsAttributeValue_r(attr);
		for(c = tmp; *c != '\0'; c++) {
			putchar((*c == ' ') ? '_' : *c);
		}
		free(tmp);
	}
}


#define SWITCHES "af:N:Lv:Vot:"

void usage() {
	printf("\nUsage: nws_search [OPTIONS] filter [Attribute ...]\n");
	printf("Query a Network Weather Service nameserver\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-N nameserver       the nameserver to be queried\n");
	printf("\t-a                  only the attributes will be displayed\n");
	printf("\t-o                  only the values will be displayed\n");
	printf("\t-f yes/no           list one attribute per line (default yes)\n");
	printf("\t-t sec              set timeout in communication\n");
	printf("\t-V                  print version\n");
	printf("\t-v level            be verbose (up to level 5)\n");
	printf("Read the man pages for example of filters that nws_search accepts.\n");
	printf("\t\t\n\n");
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n\n");
}

int
main(int argc,
     char **argv) {

	extern char *optarg;
	extern int optind;
	NwsAttribute attribute;
	int attributeArgsIndex;
	const char *filter;
	size_t filterLen;
	int formatted = 1;
	int i, timeout;
	int longestName;
	char *nameServer;
	HostSpec nsSpec;
	Object object;
	ObjectSet filteredObjects;
	int opt, verbose = 0;
	NwsAttributeCharacteristics whatToPrint = NAMES_AND_VALUES;
	char *tmp;

	nameServer = NULL;
	timeout = -1;

	while((opt = getopt(argc, argv, SWITCHES)) != EOF) {
		switch(opt) {
		case 'a':
			whatToPrint = NAMES_ONLY;
			break;

		case 'f':
			formatted = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'N':
			nameServer = strdup(optarg);
			if (nameServer == NULL) {
				ABORT("Out of memory\n");
			}
			break;

		case 'o':
			whatToPrint = VALUES_ONLY;
			break;

		case 't':
			timeout = (int)atol(optarg);
			break;

		case 'v':
			verbose = (unsigned short)atol(optarg);
			break;

		case 'V':       
			printf("nws_search for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H   
			printf(", with thread support");
#endif          
#ifdef WITH_DEBUG       
			printf(", with debug support");
#endif                  
			printf("\n\n");
			exit(0);
			break;

		default:
			usage();
			exit(1);
			break;
		}
	}

  	/* set the right print level */
	SetDiagnosticLevel(verbose, stderr, stderr);

	if (nameServer == NULL) {
		/* no Nameserver given by the user: let's try localhost */
		nameServer = strdup("localhost");
		if (nameServer == NULL) {
			ABORT("out of memory\n");
		}
	}

	/* let's talk with the nameserver */
	nsSpec = *MakeHostSpec(nameServer, DefaultHostPort(NAME_SERVER_HOST));
	if(!UseNameServer(&nsSpec)) {
		ABORT2("Unable to contact name server %s:%d\n", nsSpec.machineName, nsSpec.machinePort);
	}

	if (optind >= argc) {
		usage();
		exit(1);
	}

	filter = argv[optind];
	filterLen = strlen(filter);

	for(i = 0; i < (sizeof(SHORTCUTS) / sizeof(SHORTCUTS[0])); i++) {
		if(strncasecmp(filter, SHORTCUTS[i], filterLen) == 0) {
			filter = FILTERS[i];
			break;
		}
	}

	if(!GetObjectsTimeout(filter, &filteredObjects, timeout)) {
		exit(1);
	}

	attributeArgsIndex = optind + 1;

	ForEachObject(filteredObjects, object) {
		longestName = formatted ? LongestNameLength(object) : 0;
		if(attributeArgsIndex >= argc) {
			ForEachNwsAttribute(object, attribute) {
				PrintNwsAttribute(attribute, whatToPrint, longestName);
				if (formatted) {
					printf("\n");
				} else {
					printf(" ");
				}
			}
		} else {
			for(i = attributeArgsIndex; i < argc; i++) {
				ForEachNwsAttribute(object, attribute) {
					tmp = NwsAttributeName_r(attribute);
					if (tmp == NULL) {
						continue;
					}
					if(strcmp(argv[i], tmp) == 0) {
						PrintNwsAttribute(attribute, whatToPrint, longestName);
						if (formatted) {
							printf("\n");
						} else {
							printf(" ");
						}
					}
					free(tmp);
				}
			}
		}
		printf("\n");
	}

	FreeObjectSet(&filteredObjects);
	return 0;
}
