/** 
 * @file njamd.c 
 *
 * This object brings up NJAMD as a whole
 *
 * Copyright (C) 2001 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 */
#include <lib/njamd.h>
#include <lib/portability.h>
#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

struct nj_njamd __NJAMD__;

static void hint_main() __attribute__ ((constructor));
void __nj_njamd_fini() __attribute__ ((destructor));

/**
 * NJAMD bootstrap function
 * 
 * Called before threads, from the first malloc or pthread_create, 
 * or first entrance of any njamd public function
 */
void __nj_njamd_bootstrap_init(struct nj_njamd *njamd)
{

	if(njamd->state >= NJ_STATE_BOOTSTRAPPED)
		return;
	
	__nj_portability_bootstrap_init();

	__nj_allocator_bootstrap_init(&njamd->allocator);
	__nj_output_bootstrap_init(&njamd->output);
	__nj_prefs_bootstrap_init(&njamd->prefs);


	njamd->active_threads = FALSE;
	njamd->state = NJ_STATE_BOOTSTRAPPED;

	/* dlopen may use malloc */
	__nj_libc_syms_init(&njamd->libc_syms);
	njamd->libc_exit = __nj_libc_syms_resolve_libc(&njamd->libc_syms, "exit");

	__nj_signals_bootstrap_init(&njamd->signals, &njamd->libc_syms);
	__nj_threads_bootstrap_init(&njamd->threads, &njamd->libc_syms);

	/* So may pthreads in FBSD */
#ifdef _THREAD_SAFE
	pthread_mutex_init(&njamd->init_lock, NULL);
#endif
}

/**
 * Indirectly marks when main() starts.
 * 
 * This function guesses that by the time it was called, the program is just
 * about to enter main, since it is given a constructor attribute. It also
 * calls the NJAMD bootstrap init function.
 */
static void hint_main(void)
{
	/* Some archs (*BSD) don't call malloc before entering main at all. So we
	 * want to run primary init now before any threads exist */
	if(__NJAMD__.state == NJ_STATE_NONE)
		__nj_njamd_bootstrap_init(&__NJAMD__);
//	__nj_eprintf("Constructor run\n");

	__NJAMD__.state = NJ_STATE_CROSSED_MAIN;
}

/**
 * Makes NJAMD ready for user
 *
 * Gets prefs from the environment and propogates them all to various children
 * 
 * @param njamd The njamd object
 */
void __nj_njamd_user_init(struct nj_njamd *njamd)
{
#ifdef _THREAD_SAFE
	pthread_mutex_lock(&njamd->init_lock);
#endif
	if(njamd->state >= NJ_STATE_USER_READY)
		return;


	__nj_prefs_user_init(&njamd->prefs);

	__nj_allocator_user_init(&njamd->allocator, &njamd->libc_syms, &njamd->prefs);
	__nj_signals_user_init(&njamd->signals, &njamd->prefs);
	__nj_output_user_init(&njamd->output, &njamd->prefs);
	__nj_threads_user_init(&njamd->threads);

	__nj_portability_user_init(&njamd->prefs);

#ifndef HAVE_ATTRIB_CONSTRUCT
	atexit(__nj_njamd_fini);
#endif

	/* This MUST be last, or other threads will think that secondary init has
     * completed before it actually has. */
	njamd->state = NJ_STATE_USER_READY;

#ifdef _THREAD_SAFE
	pthread_mutex_unlock(&njamd->init_lock);
#endif

}

/**
 * Handles proper shutdown of an NJAMD program.
 * 
 * All attempts are made to ensure this function is called no matter what on
 * exit. It handles dumping of leak info, displaying stats, and syncing and
 * truncating the heap file on disk.
 */
void __nj_njamd_fini()
{
	struct nj_njamd *njamd = &__NJAMD__;
	TR;
	if(njamd->state == NJ_STATE_DESTRUCT)
		return;
	
	njamd->state = NJ_STATE_DESTRUCT;

	__nj_prefs_fini(&njamd->prefs);

//	__nj_eprintf("Shutting down NJAMD\n");
	/** @FIXME I think this can be moved to user init.. */
	__nj_libc_syms_fini(&njamd->libc_syms);
	__nj_allocator_fini(&njamd->allocator);
	__nj_signals_fini(&njamd->signals);
	__nj_output_fini(&njamd->output);
	__nj_threads_fini(&njamd->threads);

	__nj_portability_fini();
}

// vim:ts=4
