(* Copyright (C) 2004, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://helm.cs.unibo.it/
 *)

type domain_item =
 | Id of string               (* literal *)
 | Symbol of string * int     (* literal, instance num *)
 | Num of int                 (* instance num *)

(* module Domain:      Set.S with type elt = domain_item *)
module Environment:
sig
  include Map.S with type key = domain_item
  val cons: domain_item -> ('a * 'b) -> ('a * 'b) list t -> ('a * 'b) list t
  val hd: 'a list t -> 'a t

    (** last alias cons-ed will be processed first *)
  val fold_flatten: (domain_item -> 'a -> 'b -> 'b) -> 'a list t -> 'b -> 'b
end

  (** to be raised when a choice is invalid due to some given parameter (e.g.
   * wrong number of Cic.term arguments received) *)
exception Invalid_choice of Stdpp.location option * string Lazy.t

type codomain_item =
  string *  (* description *)
  (environment -> string -> Cic.term list -> Cic.term)
    (* environment, literal number, arguments as needed *)

and environment = codomain_item Environment.t

type multiple_environment = codomain_item list Environment.t

(* a simple case of extension of a disambiguation environment *)
val env_of_list:
  (string * string * Cic.term) list -> environment -> environment

val multiple_env_of_list:
  (string * string * Cic.term) list -> multiple_environment ->
    multiple_environment

module type Callbacks =
  sig

    val interactive_user_uri_choice :
      selection_mode:[`SINGLE | `MULTIPLE] ->
      ?ok:string ->
      ?enable_button_for_non_vars:bool ->
      title:string -> msg:string -> id:string -> UriManager.uri list ->
       UriManager.uri list

      val interactive_interpretation_choice :
        string -> int ->
      (Stdpp.location list * string * string) list list -> int list

    (** @param title gtk window title for user prompting
     * @param id unbound identifier which originated this callback invocation *)
    val input_or_locate_uri:
      title:string -> ?id:string -> unit -> UriManager.uri option
  end

val string_of_domain_item: domain_item -> string
val string_of_domain: domain_item list -> string

(** {3 type shortands} *)

(*
type term = CicNotationPt.term
type tactic = (term, term, GrafiteAst.reduction, string) GrafiteAst.tactic
type tactical = (term, term, GrafiteAst.reduction, string) GrafiteAst.tactical

type script_entry =
  | Command of tactical
  | Comment of CicNotationPt.location * string
type script = CicNotationPt.location * script_entry list
*)
