(* Copyright (C) 2004, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://helm.cs.unibo.it/
 *)

(* $Id: disambiguateTypes.ml 7749 2007-10-25 16:23:06Z sacerdot $ *)

(*
type term = CicNotationPt.term
type tactic = (term, term, GrafiteAst.reduction, string) GrafiteAst.tactic
type tactical = (term, term, GrafiteAst.reduction, string) GrafiteAst.tactical
type script_entry =
  | Command of tactical
  | Comment of CicNotationPt.location * string
type script = CicNotationPt.location * script_entry list
*)

type domain_item =
  | Id of string               (* literal *)
  | Symbol of string * int     (* literal, instance num *)
  | Num of int                 (* instance num *)

exception Invalid_choice of Stdpp.location option * string Lazy.t

module OrderedDomain =
  struct
    type t = domain_item
    let compare = Pervasives.compare
  end

(* module Domain = Set.Make (OrderedDomain) *)
module Environment =
struct
  module Environment' = Map.Make (OrderedDomain)

  include Environment'

  let find k env =
   match k with
      Symbol (sym,n) ->
       (try find k env
        with Not_found -> find (Symbol (sym,0)) env)
    | Num n ->
       (try find k env
        with Not_found -> find (Num 0) env)
    | _ -> find k env

  let cons k v env =
    try
      let current = find k env in
      let dsc, _ = v in
      add k (v :: (List.filter (fun (dsc', _) -> dsc' <> dsc) current)) env
    with Not_found ->
      add k [v] env

  let hd list_env =
    try
      map List.hd list_env
    with Failure _ -> assert false

  let fold_flatten f env base =
    fold
      (fun k l acc -> List.fold_right (fun v acc -> f k v acc) l acc)
      env base

end

type codomain_item =
  string *  (* description *)
  (environment -> string -> Cic.term list -> Cic.term)
    (* environment, literal number, arguments as needed *)

and environment = codomain_item Environment.t

type multiple_environment = codomain_item list Environment.t


(** adds a (name,uri) list l to a disambiguation environment e **)
let multiple_env_of_list l e = 
  List.fold_left
   (fun e (name,descr,t) -> Environment.cons (Id name) (descr,fun _ _ _ -> t) e)
   e l

let env_of_list l e = 
  List.fold_left
   (fun e (name,descr,t) -> Environment.add (Id name) (descr,fun _ _ _ -> t) e)
   e l

module type Callbacks =
  sig
    val interactive_user_uri_choice:
      selection_mode:[`SINGLE | `MULTIPLE] ->
      ?ok:string ->
      ?enable_button_for_non_vars:bool ->
      title:string -> msg:string -> id:string -> UriManager.uri list ->
      UriManager.uri list
    val interactive_interpretation_choice:
      string -> int ->
      (Stdpp.location list * string * string) list list -> int list
    val input_or_locate_uri:
      title:string -> ?id:string -> unit -> UriManager.uri option
  end

let string_of_domain_item = function
  | Id s -> Printf.sprintf "ID(%s)" s
  | Symbol (s, i) -> Printf.sprintf "SYMBOL(%s,%d)" s i
  | Num i -> Printf.sprintf "NUM(instance %d)" i

let string_of_domain dom =
  String.concat "; " (List.map string_of_domain_item dom)
