// Maria union component reference expression class -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "UnionComponent.h"
#include "UnionType.h"
#include "UnionValue.h"
#include "Valuation.h"
#include "Printer.h"

/** @file UnionComponent.C
 * Union component reference
 */

/* Copyright  1999-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

UnionComponent::UnionComponent (class Expression& expr,
				card_t i) :
  myExpr (&expr), myIndex (i)
{
  assert (myExpr && myExpr->getType ());
  assert (myExpr->getType ()->getKind () == Type::tUnion);
  const class UnionType* type =
    static_cast<const class UnionType*>(myExpr->getType ());
  assert (myExpr->isBasic ());
  setType ((*type)[myIndex]);
}

UnionComponent::~UnionComponent ()
{
  myExpr->destroy ();
}

class Value*
UnionComponent::do_eval (const class Valuation& valuation) const
{
  class Value* v = myExpr->eval (valuation);
  if (!v)
    return NULL;

  assert (v->getKind () == Value::vUnion);
  class UnionValue* u = static_cast<class UnionValue*>(v);
  if (u->getIndex () != myIndex) {
    valuation.flag (errUnion, *this);
    delete u;
    return NULL;
  }
  v = u->getValue ().copy ();
  delete u;
  return v;
}

class Expression*
UnionComponent::ground (const class Valuation& valuation,
			class Transition* transition,
			bool declare)
{
  class Expression* e = myExpr->ground (valuation, transition, declare);
  if (!e)
    return NULL;
  assert (valuation.isOK ());
  if (e == myExpr) {
    e->destroy ();
    return copy ();
  }
  else
    return static_cast<class Expression*>
      (new class UnionComponent (*e, myIndex))->ground (valuation);
}

class Expression*
UnionComponent::substitute (class Substitution& substitution)
{
  class Expression* e = myExpr->substitute (substitution);
  if (e == myExpr) {
    e->destroy ();
    return copy ();
  }
  else
    return (new class UnionComponent (*e, myIndex))->cse ();
}

bool
UnionComponent::depends (const class VariableSet& vars,
			 bool complement) const
{
  return myExpr->depends (vars, complement);
}

bool
UnionComponent::forVariables (bool (*operation)
			      (const class Expression&,void*),
			      void* data) const
{
  return myExpr->forVariables (operation, data);
}

#ifdef EXPR_COMPILE
# include "CExpression.h"

void
UnionComponent::compile (class CExpression& cexpr,
			 unsigned indent,
			 const char* lvalue,
			 const class VariableSet* vars) const
{
  char* rvalue = 0;
  class StringBuffer& out = cexpr.getOut ();
  if (cexpr.getVariable (*myExpr, rvalue))
    myExpr->compile (cexpr, indent, rvalue, vars);
  out.indent (indent);
  out.append ("if (");
  out.append (rvalue);
  out.append (".t!=");
  out.append (myIndex);
  out.append (")\n");
  cexpr.compileError (indent + 2, errUnion);
  out.indent (indent);
  out.append (lvalue);
  out.append ("=");
  out.append (rvalue);
  out.append (".u.u");
  out.append (myIndex);
  out.append (";\n");
  delete[] rvalue;
}

#endif // EXPR_COMPILE

/** Determine whether an expression needs to be enclosed in parentheses
 * @param kind	kind of the expression
 * @return	whether parentheses are necessary
 */
inline static bool
needParentheses (enum Expression::Kind kind)
{
  switch (kind) {
  case Expression::eVariable:
  case Expression::eUndefined:
  case Expression::eStructComponent:
  case Expression::eUnionComponent:
  case Expression::eVectorIndex:
    return false;
  case Expression::eCardinality:
  case Expression::eUnionType:
  case Expression::eUnion:
  case Expression::eVector:
  case Expression::eBinop:
  case Expression::eBooleanBinop:
  case Expression::eNot:
  case Expression::eRelop:
  case Expression::eBuffer:
  case Expression::eBufferRemove:
  case Expression::eBufferWrite:
  case Expression::eBufferIndex:
  case Expression::eSet:
  case Expression::eTemporalBinop:
  case Expression::eTemporalUnop:
  case Expression::eMarking:
  case Expression::eTransitionQualifier:
  case Expression::ePlaceContents:
  case Expression::eSubmarking:
  case Expression::eMapping:
  case Expression::eEmptySet:
  case Expression::eVectorAssign:
  case Expression::eVectorShift:
    assert (false);
  case Expression::eUnop:
  case Expression::eConstant:
  case Expression::eStruct:
  case Expression::eStructAssign:
  case Expression::eIfThenElse:
  case Expression::eBufferUnop:
  case Expression::eTypecast:
    break;
  }

  return true;
}

void
UnionComponent::display (const class Printer& printer) const
{
  if (::needParentheses (myExpr->getKind ())) {
    printer.delimiter ('(')++;
    myExpr->display (printer);
    --printer.delimiter (')');
  }
  else
    myExpr->display (printer);
  printer.delimiter ('.');
  printer.print (static_cast<const class UnionType*>(myExpr->getType ())
		 ->getComponentName (myIndex));
}
