// Maria typecast expression -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "Typecast.h"
#include "Type.h"
#include "LeafValue.h"
#include "Printer.h"

/** @file Typecast.C
 * Type conversion operation
 */

/* Copyright  1998-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

bool
Typecast::isCastable (const class Type& type)
{
  switch (type.getKind ()) {
  case Type::tInt:
  case Type::tCard:
  case Type::tBool:
  case Type::tChar:
  case Type::tEnum:
    return true;
  default:
    return false;
  }
}

Typecast::Typecast (const class Type& type,
		    class Expression& expr) :
  Expression (),
  myExpr (&expr)
{
  setType (type);
  assert (getType () != expr.getType ());
  assert (expr.isBasic () && !expr.isSet ());
  assert ((isCastable (*getType ()) && isCastable (*myExpr->getType ())) ||
	  Type::isCompatible (*myExpr->getType (), *getType ()));
}

Typecast::~Typecast ()
{
  myExpr->destroy ();
}

class Value*
Typecast::do_eval (const class Valuation& valuation) const
{
  class Value* v = myExpr->eval (valuation);
  if (v && !(v = v->cast (*getType ())))
    valuation.flag (errConst, *this);
  return v;
}

class Expression*
Typecast::ground (const class Valuation& valuation,
		  class Transition* transition,
		  bool declare)
{
  class Expression* e = myExpr->ground (valuation, transition, declare);
  if (!e)
    return NULL;
  assert (valuation.isOK ());
  if (e == myExpr) {
    e->destroy ();
    return copy ();
  }
  else
    return static_cast<class Expression*>
      (new class Typecast (*getType (), *e))->ground (valuation);
}

class Expression*
Typecast::substitute (class Substitution& substitution)
{
  class Expression* e = myExpr->substitute (substitution);
  if (e == myExpr) {
    e->destroy ();
    return copy ();
  }
  else
    return (new class Typecast (*getType (), *e))->cse ();
}

bool
Typecast::depends (const class VariableSet& vars,
		   bool complement) const
{
  return myExpr->depends (vars, complement);
}

bool
Typecast::forVariables (bool (*operation)
			(const class Expression&,void*),
			void* data) const
{
  return myExpr->forVariables (operation, data);
}

#ifdef EXPR_COMPILE
# include "CExpression.h"

void
Typecast::compile (class CExpression& cexpr,
		   unsigned indent,
		   const char* lvalue,
		   const class VariableSet* vars) const
{
  if (getType ()->isLeaf () && myExpr->getType ()->isLeaf ()) {
    myExpr->compile (cexpr, indent, lvalue, vars);
    if (const class Constraint* c = getType ()->getConstraint ())
      c->compileCheck (cexpr, indent, lvalue);
  }
  else {
    char* rvalue;
    if (cexpr.getVariable (*myExpr, rvalue))
      myExpr->compile (cexpr, indent, rvalue, vars);
    myExpr->getType ()->compileCast (cexpr, indent, *getType (),
				     lvalue, rvalue);
    delete[] rvalue;
  }
}

#endif // EXPR_COMPILE

/** Determine whether an expression needs to be enclosed in parentheses
 * @param kind	kind of the expression
 * @return	whether parentheses are necessary
 */
inline static bool
needParentheses (enum Expression::Kind kind)
{
  switch (kind) {
  case Expression::eVariable:
  case Expression::eConstant:
  case Expression::eUndefined:
  case Expression::eStructComponent:
  case Expression::eUnionComponent:
  case Expression::eStruct:
  case Expression::eVector:
  case Expression::eBuffer:
  case Expression::eVectorIndex:
  case Expression::eTypecast:
  case Expression::eUnop:
  case Expression::eNot:
  case Expression::eBufferUnop:
  case Expression::eTemporalUnop:
    return false;
  case Expression::eMarking:
  case Expression::eTransitionQualifier:
  case Expression::ePlaceContents:
  case Expression::eSubmarking:
  case Expression::eMapping:
  case Expression::eEmptySet:
    assert (false);
  case Expression::eUnion:
  case Expression::eUnionType:
  case Expression::eBufferRemove:
  case Expression::eBufferWrite:
  case Expression::eBufferIndex:
  case Expression::eSet:
  case Expression::eBinop:
  case Expression::eBooleanBinop:
  case Expression::eRelop:
  case Expression::eIfThenElse:
  case Expression::eTemporalBinop:
  case Expression::eCardinality:
  case Expression::eStructAssign:
  case Expression::eVectorAssign:
  case Expression::eVectorShift:
    break;
  }

  return true;
}

void
Typecast::display (const class Printer& printer) const
{
  printer.printRaw ("is");
  printer.delimiter (' ');
  if (const char* name = getType ()->getName ())
    printer.print (name);
  else
    printer.print (getType ()->getSyntacticName ());
  printer.delimiter (' ');
  if (::needParentheses (myExpr->getKind ())) {
    printer.delimiter ('(')++;
    myExpr->display (printer);
    --printer.delimiter (')');
  }
  else
    myExpr->display (printer);
}
