/*                                                                          
        Source: check_boot.c
        Author: Adam G. Bowen (agbowen@bealenet.com)
        $Revision: 1.5 $
        $Date: 1999/05/17 14:35:26 $
 
 * Program: System uptime plugin for NetSaint
 * License: GPL
 * Copyright (c) 1999 Adam G. Bowen (agbowen@bealenet.com)
 *
 * Description:
 *
 * This plugin will check the system uptime on the remote host at ipaddress and
 * generate an alert if the uptime is above the <max_up> value or if
 * (<weekend_boot> = Y) and ((max_up - uptime) < 7) and ((today = Saturday) or
 * (today = Sunday)).
 *
 * Other errors result in a STATE_UNKNOWN return.
 *
 * Command line:
 *
 * check_boot ipaddress sys_type
 * check_boot ipaddress sys_type <weekend_boot>
 * check_boot ipaddress sys_type <weekend_boot> <max_up>
 *
 * Required input:
 *
 * ipaddress = The ipaddress of the remote system to run the check on.
 * sys_type = The remote system type.
 *
 * Optional input:
 *
 * <weekend_boot> = Should this system only be rebooted on the weekend.
 * <max_up> = Number of days up required to return a CRITICAL state.
 *
 * Notes:
 *
 * If <weekend_boot> and <max_up> are not passed on the command line, they will
 * be set to the default values in the check_boot config file.
 *
 * If <weekend_boot> is passed on the command line, and <max_up> is not,
 * <max_up> will be set to the default value in the check_boot config file.
 *
 * sys_type is used to determine which config file to use to generate the
 * remote command.
 *
 * The configuration file /usr/local/netsaint/config/check_boot/local
 * contains the following values:
 *
 * RSH_COMMAND|<location of rsh command on netsaint system>|
 * CUT_COMMAND|<location of cut command on netsaint system>|
 * WEEKEND_BOOT|<should check system only be booted on the weekend>|
 * MAX_UP|<max days between reboot>|
 *
 * The configuration file /usr/local/netsaint/config/check_boot/<sys_type>
 * contains the following values:
 *
 * UPTIME_COMMAND|<location of uptime command on system sys_type>|
 * CUT_OPTIONS|<options for cut command to retrieve required info>|
 *
 * $Log: check_boot.c,v $
 * Revision 1.5  1999/05/17 14:35:26  netsaint
 * Changed the plugin to use a seperate config directory.  This directory is the
 * CONFIG_DIR/command_name directory.
 *
 * Revision 1.4  1999/05/14 03:01:17  netsaint
 * Added the following integer variable:
 * 	socket_name
 * Changed the call check_net to open_socket.  Added a call to recv_socket and
 * close_socket.  The check_net subroutine was changed to provide more
 * flexibility.
 * Added a call to the subroutine get_command_name.
 * Changed the error checking routines to ensure that any error cause the
 * program to terminate.
 *
 * Revision 1.3  1999/05/07 15:34:59  netsaint
 * Removed the char variable error_buffer and the FILE *error_fp variable.
 * These variables are no longer needed since the printing of the error file is
 * handled in a subroutine.
 * Added a call to the check_output_file subroutine.  This routine checks the
 * status of the output file.  Also removed the struct stat file_stat variable.
 *
 * Revision 1.2  1999/05/03 14:48:17  netsaint
 * Changed the config_file_prefix to config_file_fs_prefix.  Added the following
 * character variables:
 * 	config_file_net_prefix
 * 	config_file_net
 * 	expected
 * 	protocol
 * Added the following integer variables:
 * 	result
 * 	telnet_port
 * Added signal handler for the SIGALRM signal.  All these changes were to
 * prevent the plugin from hanging when attempting to perform check on a remote
 * system that is down or not working properly.  Prior to issuing the rsh
 * command, the plugin will attempt to establish a telnet session to the
 * plugin will not issue the rsh command.  Prior to establishing the telnet
 * connection, an alarm is set.  If the telnet connections does not return
 * control to the plugin before the timer expires, a SIGALRM signal will be
 * sent to the process which will caues the plugin to exit with a
 * STATE_CRITICAL error.
 * Added struct stat file_stat.
 * This is used to get the size of the out_put_file.  If the size is zero,
 * the plugin exits with a STATE_UNKNOWN.  Also change the error_file printing
 * to a subroutine in the plugins.h file.
 *
 * Revision 1.1  1999/04/28 15:18:06  netsaint
 * Initial revision
 *
*/

#include "/usr/local/src/netsaint/include/plugins.h"

int main(int argc, char *argv[])
{
  char expected[MAX_CHARS]="";
  char token_sep[] = " ,";
  char weekend_boot[MAX_CHARS]="";

  char command_line[MAX_CHARS];
  char command_name[MAX_CHARS];
  char config_file_local[MAX_CHARS];
  char config_file_net[MAX_CHARS];
  char config_file_remote[MAX_CHARS];
  char cut_command[MAX_CHARS];
  char cut_options[MAX_CHARS];
  char day_of_week[MAX_CHARS];
  char error_file[MAX_CHARS];
  char input_buffer[MAX_CHARS];
  char ip_address[MAX_CHARS];
  char out_put_file[MAX_CHARS];
  char port_telnet[MAX_CHARS];
  char protocol[MAX_CHARS];
  char rsh_command[MAX_CHARS];
  char system_name[MAX_CHARS];
  char temp_value[MAX_CHARS];
  char uptime_command[MAX_CHARS];
  char uptime_return[MAX_CHARS];
  char up_days[MAX_CHARS];
  char up_max[MAX_CHARS];
  char up_units[MAX_CHARS];

  FILE *out_put_fp;

  int boot_weekend;
  int days_up;
  int get_max_up;
  int get_weekend;
  int is_weekend;
  int max_up;
  int result;
  int return_value;
  int socket_name;
  int telnet_port;
  int today;

  struct tm *curtime;
  
  time_t bintime;

  /* Initialize alarm signal handling */

  signal(SIGALRM,alarm_signal);

  strcpy(command_name,get_command_name(argv[0]));
  if(!((argc==3) || (argc==4) || (argc==5)))
  {
    printf("\n");
    printf(" Incorrect number of arguments supplied\n");
    printf("\n");
    printf(" System uptime plugin for NetSaint\n");
    printf(" Copyright (c) 1999 Adam G. Bowen (agbowen@bealenet.com)\n");
    printf(" $Revision: 1.5 $\n");
    printf(" Last Modified $Date: 1999/05/17 14:35:26 $\n");
    printf(" License: GPL\n");
    printf("\n");
    printf(" Description:\n");
    printf("\n");
    printf("This plugin will check the system uptime on the remote host at ipaddress and\n");
    printf("generate an alert if the uptime is above the <max_up> value or if\n");
    printf("(<weekend_boot> = Y) and ((max_up - uptime) < 7) and ((today = Saturday) or\n");
    printf("(today = Sunday)).\n");
    printf("\n");
    printf(" Usage: %s ipaddress sys_type\n",command_name);
    printf(" Usage: %s ipaddress sys_type <weekend_boot>\n",command_name);
    printf(" Usage: %s ipaddress sys_type <weekend_boot> <max_up>\n",command_name);
    printf("\n");
    printf(" Required input:\n");
    printf("\n");
    printf(" ipaddress = The ipaddress of the remote system to run the check on.\n");
    printf(" sys_type = The remote system type.\n");
    printf("\n");
    printf(" Optional input:\n");
    printf("\n");
    printf(" <weekend_boot> = Should this system only be rebooted on the weekend?\n");
    printf(" <max_up> = Number of days up required to result in a CRITICAL state.\n");
    printf("\n");
    printf(" If <weekend_boot> and <max_up> are not passed on the command line, they will\n");
    printf(" be set to the default values in the %s config file.\n", command_name);
    printf("\n");
    printf(" If <weekend_boot> is passed on the command line, and <max_up> is not,\n");
    printf(" <max_up> will be set to the default value in the %s config file.\n",command_name);
    printf("\n");
    printf(" sys_type is used to determine which config file to use to generate the\n");
    printf(" remote command.\n");
    printf("\n");
    return_value = STATE_UNKNOWN;
  }
  else
  {
    /* Set up config files and get the command line information */

    strcpy(ip_address,argv[1]);
    strcpy(system_name,argv[2]);

    strcpy(config_file_local,CONFIG_DIR);
    strcpy(config_file_remote,CONFIG_DIR);
    strcpy(config_file_net,CONFIG_DIR);
    strcat(config_file_local,command_name);
    strcat(config_file_remote,command_name);
    strcat(config_file_net,CHECK_TELNET);
    strcat(config_file_local,"/local");
    strcat(config_file_remote,"/");
    strcat(config_file_remote,system_name);
    strcat(config_file_net,"/");
    strcat(config_file_net,system_name);

    if(argc == 3)
    {
      get_max_up = TRUE;
      get_weekend = TRUE;
    }
    else if(argc == 4)
    {
      get_max_up = TRUE;
      get_weekend = FALSE;
      strupr(strncpy(weekend_boot,argv[3],1));
    }
    else
    {
      get_max_up = FALSE;
      get_weekend = FALSE;
      strupr(strncpy(weekend_boot,argv[3],1));
      strcpy(up_max,argv[4]);
    }

    /* Check if config files exist */

    if (access(config_file_local, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_local);
      return_value = STATE_UNKNOWN;
    }
    else if (access(config_file_remote, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_remote);
      return_value = STATE_UNKNOWN;
    }
    else if (access(config_file_net, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_net);
      return_value = STATE_UNKNOWN;
    }
    else
    {

      /* Local config file variables */

      if((get_weekend == TRUE) && ((return_value=get_var("WEEKEND_BOOT", config_file_local, weekend_boot)) != STATE_OK))
      {
        printf("WEEKEND_BOOT entry not found in config file %s!\n",config_file_local);
      }
      else if((get_max_up == TRUE) && ((return_value=get_var("MAX_UP", config_file_local, up_max)) != STATE_OK))
      {
        printf("MAX_UP entry not found in config file %s!\n",config_file_local);
      }
      else if((return_value=get_var("RSH_COMMAND", config_file_local, rsh_command)) != STATE_OK)
      {
        printf("RSH_COMMAND entry not found in config file %s!\n", config_file_local);
      }
      else if((return_value=get_var("CUT_COMMAND", config_file_local, cut_command)) != STATE_OK)
      {
        printf("CUT_COMMAND entry not found in config file %s!\n", config_file_local);
      }

      /* Remote config file variables */

      else if((return_value=get_var("UPTIME_COMMAND", config_file_remote, uptime_command)) != STATE_OK)
      {
        printf("UPTIME_COMMAND entry not found in config file %s!\n", config_file_remote);
      }
      else if((return_value=get_var("CUT_OPTIONS", config_file_remote, cut_options)) != STATE_OK)
      {
        printf("CUT_OPTIONS entry not found in config file %s!\n",config_file_remote);
      }

      /* Network config file variables */

      else if((return_value=get_var("TELNET_PORT", config_file_net, port_telnet)) != STATE_OK)
      {
        printf("TELNET_PORT entry not found in config file %s!\n",config_file_net);
      }
      else if((return_value=get_var("TELNET_PROTO", config_file_net, protocol)) != STATE_OK)
      {
        printf("TELNET_PROTO entry not found in config file %s!\n",config_file_net);
      }
      else
      {

        /* Check alert level consistency */

        max_up=atoi(up_max);
        if((strcmp(weekend_boot,"Y") == 0) && (max_up < 7))
        {
          printf("Inconsistency in parameters: weekend_boot set to yes and max_up (%d days) less than 7 days.\n",max_up);
          return_value = STATE_UNKNOWN;
        }
        else if(max_up < 1)
        {
          printf("Invalid parameter: max_up (%d) less than 1 day.\n",max_up);
          return_value = STATE_UNKNOWN;
        }
        else
        {
          return_value = STATE_OK;
        }
        if(return_value == STATE_OK)
        {

          /* Check the network */

          telnet_port=atoi(port_telnet);
          alarm(TIME_OUT);
          if((result=open_socket(&socket_name, ip_address, telnet_port, protocol)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,telnet_port);
          }
          else if((result=recv_socket(&socket_name, expected)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,telnet_port);
          }
          else if((result=close_socket(&socket_name)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,telnet_port);
          }
          else
          {
            alarm(0);

            /* Generate a out_put and error file names */

            strcpy(out_put_file, tmpnam(NULL));
            strcpy(error_file, tmpnam(NULL));

            /* set the command line and arguments to use for the check */
  
            sprintf(command_line,"%s %s %s </dev/null 2>%s|%s %s >%s 2>>%s",rsh_command, ip_address, uptime_command, error_file, cut_command, cut_options, out_put_file, error_file);

            /* Run the command */
  
            system(command_line);

            return_value=check_output_file(out_put_file);
            if(return_value != STATE_OK)
            {
              print_error(error_file);
            }
            else
            {

              out_put_fp=fopen(out_put_file,"r");
      
              /* Retrive single line from output file */

              fgets(input_buffer,MAX_CHARS-1,out_put_fp);

              /* Populate variables */
      
              strcpy(temp_value,strtok(input_buffer,token_sep));
              strcpy(temp_value,strtok(NULL,token_sep));
              strcpy(up_days,strtok(NULL,token_sep));
              if(strchr(up_days,':') == NULL)
              {
                strupr(strcpy(up_units,strtok(NULL,token_sep)));
                if(strstr(up_units,"DAY"))
                {
                  days_up = atoi(up_days);
                  time(&bintime);
                  curtime = localtime(&bintime);
                  strftime(day_of_week,MAX_CHARS,"%w",curtime);
                  today = atoi(day_of_week);
              
                  if((today == SATURDAY) || (today == SUNDAY))
                  {
                    is_weekend = TRUE;
                  }
                  else
                  {
                    is_weekend = FALSE;
                  }
                  if(strcmp(weekend_boot,"Y") == 0)
                  {
                    boot_weekend = TRUE;
                  }
                  else
                  {
                    boot_weekend = FALSE;
                  }
                  if(days_up > max_up)
                  {
                    return_value=STATE_CRITICAL;
                  }
                  else if ((boot_weekend == TRUE) && (is_weekend == TRUE) && ((max_up - days_up) < 7))
                  {
                    return_value=STATE_WARNING;
                  }
                  else
                  {
                    return_value=STATE_OK;
                  }
                }
                else
                {
                  return_value=STATE_OK;
                }
                strcpy(uptime_return,up_days);
                strcat(uptime_return," ");
                strcat(uptime_return,up_units);
              }
              else
              {
                return_value=STATE_OK;
                sprintf(uptime_return,"%s",up_days);
              }

              /* Close output file */

              fclose(out_put_fp);
              if(return_value==STATE_OK)
              {
                printf("System uptime ok - up %s.\n", uptime_return);
              }
              else if (return_value==STATE_WARNING)
              {
                printf("System uptime waring - system up %s, will exceed max up (%d days) before next week end.\n", uptime_return, max_up);
              }
              else
              {
                printf("System uptime error - system up %s, exceeded max up (%d days).\n", uptime_return, max_up);
              }
            }

            /* remove the output and error files */
  
            remove(out_put_file);
            remove(error_file);
          }
        }
      }
    }
  }
  return return_value;
}
