/*
 *   $Id: check_spread.tar,v 1.1 2002/04/05 22:19:29 sghosh Exp $
 * 
 * "check_spread" plugin for Nagios (aka Netsaint)
 * by John David Duncan <jdd@efn.org>
 *
 * Nagios is an automated network monitoring application.
 * For more information about Nagios, see http://www.nagios.org/
 *
 * The Spread Toolkit provides application-level multicast 
 * and group communication support. 
 * For more information about Spread, see http://www.spread.org/
 *
 * This plugin allows Nagios to monitor a spread server.
 * The program simply attempts to SP_connect() to a server, and then
 * exits with a value of 0 on success or non-zero to report failure.
 * 
 * License Information:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* a spread header file: */
#include "sp.h"


/* For plugins distributed with Netsaint/Nagios,
   these values are defined in plugins/common.h --
   however, check_spread can stand alone (without Netsaint),
   so that file is not required. */
#ifndef STATE_CRITICAL
 #define STATE_CRITICAL   2
 #define STATE_WARNING    1
 #define STATE_OK         0
 #ifdef POSIX_STATE_DEFS
  #define STATE_UNKNOWN    3
 #else
  #define STATE_UNKNOWN   -1
 #endif
#endif 


#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

int		usage();
int		exit_with();
int 	check_spread();
int 	set_timeout();

static char const rcsid[] = "$Revision: 1.1 $";
static int DEBUG=0;
enum	{ NAME_LENGTH = 80 };
char	User[NAME_LENGTH];
char    Spread_name[NAME_LENGTH];
char    Private_group[MAX_GROUP_NAME];
mailbox Mbox;
sp_time	warn_timeout, crit_timeout;

extern	char 	*optarg;


int main( int argc, char *argv[] ) {
    int	try1, try2;
	int ch;

    strcpy(User, "");
    sprintf(Spread_name, "4803@localhost");
    warn_timeout.sec = 0;
    warn_timeout.usec = 500000;
    crit_timeout.sec = 4;
    crit_timeout.usec = 0;
    
	while ((ch=getopt(argc,argv,"vhs:u:w:c:")) != -1)
		switch(ch) {
			case 'v':
				DEBUG=1;
				break;
			case 's':
				strncpy(Spread_name, optarg, NAME_LENGTH);
				break;
			case 'u':
				strncpy(User, optarg, NAME_LENGTH);
				break;
			case 'w':
				if(!(set_timeout(&warn_timeout,optarg)))
					return usage();
				break;
			case 'c':
				if(!(set_timeout(&crit_timeout,optarg)))
					return usage();
				break;
			case 'h':
			default:
				return usage();
	}
	
    try1 = check_spread(warn_timeout);
    if(try1 == ACCEPT_SESSION)
        return exit_with(STATE_OK);
    else {
        SP_error(try1);		/* print spread's error message */
        switch(try1) {
            case REJECT_VERSION:	/* These errors probably indicate */
            case ILLEGAL_SPREAD:	/* that the spread daemon */ 
            case REJECT_ILLEGAL_NAME:	/* is actually running. */
            case REJECT_NOT_UNIQUE:
                return exit_with(STATE_UNKNOWN);
        }
        /* The error from try1 was CONNECTION_CLOSED or COULD_NOT_CONNECT */    
        /*  Try again with the longer timeout: */
        try2=check_spread( crit_timeout );
        if (try2 == ACCEPT_SESSION) return exit_with(STATE_WARNING);
        else if (try2 != try1) SP_error(try2);
        return exit_with(STATE_CRITICAL);
    }
}


int set_timeout(sp_time *timeout, char *sec) {
	double fl_secs;

	fl_secs=atof(sec);
	if(errno == ERANGE || errno == EDOM) {
		perror("timeout value");
		return 0;
	}
	timeout->sec  = (int) fl_secs;
	timeout->usec = (int) ( 1000000 * ( fl_secs - (int) fl_secs) );  /* microseconds */

	return 1;
}


int check_spread(sp_time timeout) {
    int ret;
	if(DEBUG)
		printf("Checking spread daemon %s; User: %s, timeout: %d sec., %d usec. \n",
				Spread_name, User, timeout.sec, timeout.usec);

    ret=SP_connect_timeout( Spread_name, User, 0, 1, &Mbox, Private_group, timeout );
    if (ret == ACCEPT_SESSION) {
        printf("Connected to spread server %s\n",Spread_name);
        SP_disconnect( Mbox );
    }
    return ret;
}


int exit_with(int x) {
    switch(x) {
        case STATE_OK:
            puts("OK.");
            break;
        case STATE_WARNING:
            puts("WARNING. Connection succeeded only on second attempt.");
			break;
        case STATE_UNKNOWN:
            puts("UNKNOWN. Cannot determine state of spread daemon.");
			break;
        case STATE_CRITICAL:
            puts("CRITICAL. Connection failed.");
        }
    return x;
}
          
			  
int usage() {
	char seconds[10];

    printf("\ncheck_spread (%s) -- check the status of a spread daemon.\n\n",rcsid);
    printf("Usage: \n");
	printf("\t[-v]              : be verbose\n");
	printf("\t[-h]              : show this usage description\n");
    printf("\t[-s <address>]    : spread address [%s]\n",Spread_name);
    printf("\t[-u <user name>]  : spread user name [none]\n");
	sprintf(seconds,"%d.%d",warn_timeout.sec, (warn_timeout.usec / 100000));
    printf("\t[-w <seconds>]    : connection timeout for warning [%s]\n",seconds); 
	sprintf(seconds,"%d.%d",crit_timeout.sec, (crit_timeout.usec / 100000));
    printf("\t[-c <seconds>]    : connection timeout for failure [%s]\n",seconds);
    printf("\nExits reporting OK, UNKNOWN, WARNING, or CRITICAL.\n");
	return STATE_UNKNOWN;
}

