// File: check_memory.c
// Author: Joshua Jackson <jjackson@vortech.net>
// Date: 08/19/1999
//
// Purpose: plugin for NetSaint that allows monitoring of available
//          system memory.  Uses (free_ram + free_swap + buffers + cache)
//          as a base caclulation for the amount of available memory.
//
// Addition routines stolen from the procps library

#include <stdio.h>
#include <stdlib.h>
#include <linux/version.h>
#include "sysinfo.h"
#include "version.h"

#define STATE_CRITICAL 2
#define STATE_WARNING 1
#define STATE_OK 0
#define STATE_UNKNOWN -1
#define MSG_OK "OK"
#define MSG_WARNING "WARNING"
#define MSG_CRITICAL "CRITICAL"

void usage()
{
	fprintf(stderr, "Usage: check_memory <WrnLvl> <CritLvl> \n\n");
	fprintf(stderr, "  <WrnLvl>   %% of memory free to trigger a warning\n");
	fprintf(stderr, "  <CritLvl>  %% of memory free to trigger a critical error\n\n");
	fprintf(stderr, "The calculation is based on physical and swap.\n");
	exit(STATE_UNKNOWN);
}

int main(int argc, char **argv)
{
	unsigned **mem;

	unsigned memtotal, memfree;
	unsigned pfree;
	unsigned warnlvl, critlvl;
	char *statmsg;
	int errlvl;

	if (!(mem = meminfo()) || mem[meminfo_main][meminfo_total] == 0) {
		fprintf(stderr, "ERROR: Cannot read from /proc/meminfo\n");
		exit(STATE_CRITICAL);
	}

	if (argc != 3) {
		fprintf(stderr, "Only %i parameters specified\n", argc); 
		usage();
	}

	argv++;
	warnlvl = atoi(*argv);
	argv++;
	critlvl = atoi(*argv);

	if (warnlvl <= critlvl) {
		fprintf(stderr, "ERROR: Warning level must be greater than critical level!\n");
		exit(STATE_UNKNOWN);
	}

	memtotal = (mem[meminfo_main][meminfo_total] + mem[meminfo_swap][meminfo_total]) >> 10;
	memfree = (mem[meminfo_main][meminfo_free] + mem[meminfo_swap][meminfo_free]) >> 10;
	memfree += (mem[meminfo_main][meminfo_cached] + mem[meminfo_swap][meminfo_buffers]) >> 10;


	pfree = (memfree * 100) / memtotal;

	if (pfree > warnlvl) {
		errlvl = STATE_OK;
		statmsg = MSG_OK;
	} else if ((pfree <= warnlvl) && (pfree > critlvl)) {
		errlvl = STATE_WARNING;
		statmsg = MSG_WARNING;
	} else if (pfree <= critlvl) {
		errlvl = STATE_CRITICAL;
		statmsg = MSG_CRITICAL;
	}
	fprintf(stdout, "Status: %s - %i%% memory free.\n", statmsg, pfree); 

	return(errlvl);
}
