/*
 * drivers/char/genrtc.c -- generic dummy /dev/rtc
 *
 * started 11/12/1999 Sam Creasey
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 * 
 * portions of this code based on rtc.c, which is
 * Copyright (C) 1996 Paul Gortmaker.  See that file for more information.
 */


#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/fcntl.h>
#include <linux/mc146818rtc.h>
#include <linux/init.h>
#include <linux/kd.h>


#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#ifdef mc68000
#include <asm/machdep.h>
#endif

#ifdef __hppa__
#include <asm/pdc.h>
#endif

#define GEN_RTC_VERSION "v1.02 05/27/1999"

/* make sure only one process opens /dev/rtc */
static unsigned char gen_rtc_opened = 0;

/* uie emulation -- basically a 1 second timer */
static unsigned char gen_uie_on = 0;
static u_int32_t gen_uie_data = 0;
static struct timer_list gen_uie_timer;
static DECLARE_WAIT_QUEUE_HEAD(gen_rtc_wait);

/* archetecture specific time acquisition functions */
#ifdef mc68000
static int get_hw_time(struct rtc_time *wtime)
{
	struct hwclk_time hwtime;

	mach_hwclk(0, &hwtime);
	wtime->tm_sec = hwtime.sec;
	wtime->tm_min = hwtime.min;
	wtime->tm_hour = hwtime.hour;
	wtime->tm_mday = hwtime.day;
	wtime->tm_mon = hwtime.mon;
	wtime->tm_year = hwtime.year;

	return 0;
}

static int set_hw_time(struct rtc_time *wtime)
{

	struct hwclk_time hwtime;
	
	hwtime.sec = wtime->tm_sec;
	hwtime.min = wtime->tm_min;
	hwtime.hour = wtime->tm_hour;
	hwtime.day = wtime->tm_mday;
	hwtime.mon = wtime->tm_mon;
	hwtime.year = wtime->tm_year;
	
	mach_hwclk(1, &hwtime);

	return 0;
}
#endif /* mc68000 */

/* PARISC versions */
#ifdef __hppa__

#define SECS_PER_HOUR   (60 * 60)
#define SECS_PER_DAY    (SECS_PER_HOUR * 24)

# define __isleap(year) \
  ((year) % 4 == 0 && ((year) % 100 != 0 || (year) % 400 == 0))


/* How many days come before each month (0-12).  */
static const unsigned short int __mon_yday[2][13] =
{
	/* Normal years.  */
	{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
	/* Leap years.  */
	{ 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }
};

static int get_hw_time(struct rtc_time *wtime)
{
	struct pdc_tod tod_data;
	long int days, rem, y;
	const unsigned short int *ip;

	if(pdc_tod_read(&tod_data) < 0)
		return -1;

	
	// most of the remainder of this function is:
//	Copyright (C) 1991, 1993, 1997, 1998 Free Software Foundation, Inc.
//	This was originally a part of the GNU C Library.
//      It is distributed under the GPL, and was swiped from offtime.c


	days = tod_data.tod_sec / SECS_PER_DAY;
	rem = tod_data.tod_sec % SECS_PER_DAY;

	wtime->tm_hour = rem / SECS_PER_HOUR;
	rem %= SECS_PER_HOUR;
	wtime->tm_min = rem / 60;
	wtime->tm_sec = rem % 60;

	y = 1970;
	
#define DIV(a, b) ((a) / (b) - ((a) % (b) < 0))
#define LEAPS_THRU_END_OF(y) (DIV (y, 4) - DIV (y, 100) + DIV (y, 400))

	while (days < 0 || days >= (__isleap (y) ? 366 : 365))
	{
		/* Guess a corrected year, assuming 365 days per year.  */
		long int yg = y + days / 365 - (days % 365 < 0);

		/* Adjust DAYS and Y to match the guessed year.  */
		days -= ((yg - y) * 365
			 + LEAPS_THRU_END_OF (yg - 1)
			 - LEAPS_THRU_END_OF (y - 1));
		y = yg;
	}
	wtime->tm_year = y - 1900;

	ip = __mon_yday[__isleap(y)];
	for (y = 11; days < (long int) ip[y]; --y)
		continue;
	days -= ip[y];
	wtime->tm_mon = y;
	wtime->tm_mday = days + 1;
	
	return 0;
}

static int set_hw_time(struct rtc_time *wtime)
{
	u_int32_t secs;

	secs = mktime(wtime->tm_year + 1900, wtime->tm_mon + 1, wtime->tm_mday, 
		      wtime->tm_hour, wtime->tm_min, wtime->tm_sec);

	if(pdc_tod_set(secs, 0) < 0)
		return -1;
	else
		return 0;

}

#endif /* __hppa__ */

/* handle timer hit -- basically acts like we got a uie */
static void gen_do_uie_timer(unsigned long t)
{
	gen_uie_data = 1;
	wake_up_interruptible(&gen_rtc_wait);
	mod_timer(&gen_uie_timer, jiffies + HZ);

}
	

/* file operations, some derived from rtc.c */

static long long gen_rtc_llseek(struct file *file, loff_t offset, int origin)
{
        return -ESPIPE;
}

/* just like the normal rtc... */

static ssize_t gen_rtc_read(struct file *file, char *buf,
                        size_t count, loff_t *ppos)
{
        DECLARE_WAITQUEUE(wait, current);
        unsigned long data;
        ssize_t retval;

        if (count != sizeof (unsigned int) && count != sizeof (unsigned long))
                return -EINVAL;

        add_wait_queue(&gen_rtc_wait, &wait);

        current->state = TASK_INTERRUPTIBLE;

        while ((data = xchg(&gen_uie_data, 0)) == 0) {
                if (file->f_flags & O_NONBLOCK) {
                        retval = -EAGAIN;
                        goto out;
                }
                if (signal_pending(current)) {
                        retval = -ERESTARTSYS;
                        goto out;
                }
                schedule();
        }
	/* first test allows optimizer to nuke this case for 32-bit machines */
	if (sizeof (int) != sizeof (long) && count == sizeof (unsigned int)) {
		unsigned int uidata = data;
		retval = put_user(uidata, (unsigned long *)buf);
	}
	else {
		retval = put_user(data, (unsigned long *)buf);
	}
        if (!retval)
                retval = sizeof(unsigned long);
 out:
        current->state = TASK_RUNNING;
        remove_wait_queue(&gen_rtc_wait, &wait);

        return retval;
}

/* handle ioctls -- many of them are not implemented */

static int gen_rtc_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
                     unsigned long arg)
{
	struct rtc_time wtime;

	switch(cmd) {
	case RTC_RD_TIME:
		/* ignore wday, yday, isdst */
		if(get_hw_time(&wtime) == -1)
			return -EFAULT;
		else
			return copy_to_user((void *)arg, &wtime, sizeof(wtime)) 
				? -EFAULT : 0;
	
	case RTC_SET_TIME:
		if(copy_from_user(&wtime, (struct rtc_time *)arg, 
				  sizeof(struct rtc_time)))
			return -EFAULT;

		if(set_hw_time(&wtime) == 0)
			return 0;
		else
			return -EFAULT;

	case RTC_UIE_ON:
		/* set a timer to fake clock ticks every second */
		if(gen_uie_on) 
			return 0;

		gen_uie_on = 1;
		gen_uie_timer.expires = jiffies + HZ;
		add_timer(&gen_uie_timer);
		
		return 0;

	case RTC_UIE_OFF:
		if(!gen_uie_on)
			return 0;
		
		gen_uie_on = 0;
		del_timer(&gen_uie_timer);
		return 0;
		
	default:
		break;
	}

	return -EINVAL;
}

static int gen_rtc_open(struct inode *inode, struct file *file)
{
	if(gen_rtc_opened)
		return -EBUSY;

	gen_uie_data = 0;
	gen_rtc_opened = 1;
	return 0;
}

static int gen_rtc_release(struct inode *inode, struct file *file)
{
	if(gen_uie_on) {
		gen_uie_on = 0;
		del_timer(&gen_uie_timer);
	}

	gen_rtc_opened = 0;
	
	return 0;
}
		
/*
 *      The various file operations we support.
 */

static struct file_operations gen_rtc_fops = {
	llseek: gen_rtc_llseek,
	read: gen_rtc_read,
	ioctl: gen_rtc_ioctl,
	open: gen_rtc_open,
	release: gen_rtc_release,
};

/* misc device stuff */

static struct miscdevice gen_rtc_dev=
{
        RTC_MINOR,
        "rtc",
        &gen_rtc_fops
};

int __init rtc_generic_init(void)
{
	
	printk("Generic RTC Driver %s Sam Creasey (sammy@oh.verio.com)\n", GEN_RTC_VERSION);

	init_timer(&gen_uie_timer);
	gen_uie_timer.function = gen_do_uie_timer;

	misc_register(&gen_rtc_dev);


	return 0;
}
	
module_init(rtc_generic_init);
