/*
 * Copyright (C) 2007 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: vfsub.h,v 1.15 2007/07/30 04:12:54 sfjro Exp $ */

#ifndef __AUFS_VFSUB_H__
#define __AUFS_VFSUB_H__

#ifdef __KERNEL__

#include <linux/fs.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18)
#include <linux/uaccess.h>
#else
#include <asm/uaccess.h>
#endif

/* ---------------------------------------------------------------------- */

/* lock subclass for hidden inode */
/* default MAX_LOCKDEP_SUBCLASSES(8) is not enough */
// todo: reduce it by dcsub.
enum {
	AuLsc_I_Begin = I_MUTEX_QUOTA,
	AuLsc_I_GPARENT,	/* setattr with inotify */
	AuLsc_I_PARENT,		/* hidden inode, parent first */
	AuLsc_I_CHILD,
	AuLsc_I_PARENT2,	/* copyup dirs */
	AuLsc_I_CHILD2,
	AuLsc_I_End
};

/* simple abstraction */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 16)
static inline void vfsub_i_lock(struct inode *i)
{
	down(&i->i_sem);
}

static inline void vfsub_i_lock_nested(struct inode *i, unsigned lsc)
{
	vfsub_i_lock(i);
}

static inline void vfsub_i_unlock(struct inode *i)
{
	up(&i->i_sem);
}

static inline int vfsub_i_trylock(struct inode *i)
{
	return down_trylock(&i->i_sem);
}

#define IMustLock(i)	AuDebugOn(!down_trylock(&(i)->i_sem))
#else
static inline void vfsub_i_lock(struct inode *i)
{
	mutex_lock(&i->i_mutex);
}

static inline void vfsub_i_lock_nested(struct inode *i, unsigned lsc)
{
	mutex_lock_nested(&i->i_mutex, lsc);
}

static inline void vfsub_i_unlock(struct inode *i)
{
	mutex_unlock(&i->i_mutex);
}

static inline int vfsub_i_trylock(struct inode *i)
{
	return mutex_trylock(&i->i_mutex);
}

#define IMustLock(i)	MtxMustLock(&(i)->i_mutex)
#endif /* LINUX_VERSION_CODE */

/* ---------------------------------------------------------------------- */

#ifdef CONFIG_AUFS_WORKAROUND_FUSE
int au_update_fuse_h_inode(struct vfsmount *h_mnt, struct dentry *h_dentry,
			   int dlgt);
#else
static inline
int au_update_fuse_h_inode(struct vfsmount *h_mnt, struct dentry *h_dentry,
			   int dlgt)
{
	return 0;
}
#endif

/* simple abstractions, for future use */
static inline
int do_vfsub_permission(struct inode *inode, int mask, struct nameidata *nd)
{
	LKTRTrace("i%lu, mask 0x%x, nd %p\n", inode->i_ino, mask, nd);
	return permission(inode, mask, nd);
}

static inline
struct file *vfsub_filp_open(const char *path, int oflags, int mode)
{
	struct file *err;

	LKTRTrace("%s\n", path);

	lockdep_off();
	err = filp_open(path, oflags, mode);
	lockdep_on();
	if (!err) {
		int e;
		e = au_update_fuse_h_inode(err->f_vfsmnt, err->f_dentry,
					   /*dlgt*/0);
		AuDebugOn(e);
	}
	return err;
}

static inline
int vfsub_path_lookup(const char *name, unsigned int flags,
		      struct nameidata *nd)
{
	int err;

	LKTRTrace("%s\n", name);

	/* lockdep_off(); */
	err = path_lookup(name, flags, nd);
	/* lockdep_on(); */
	if (!err)
		err = au_update_fuse_h_inode(nd->mnt, nd->dentry, /*dlgt*/0);
	return err;
}

/* ---------------------------------------------------------------------- */

static inline
int do_vfsub_create(struct inode *dir, struct dentry *dentry, int mode,
		    struct nameidata *nd)
{
	LKTRTrace("i%lu, %.*s, 0x%x\n", dir->i_ino, DLNPair(dentry), mode);
	return vfs_create(dir, dentry, mode, nd);
}

static inline
int do_vfsub_symlink(struct inode *dir, struct dentry *dentry,
		     const char *symname, int mode)
{
	LKTRTrace("i%lu, %.*s, %s, 0x%x\n",
		  dir->i_ino, DLNPair(dentry), symname, mode);
	return vfs_symlink(dir, dentry, symname, mode);
}

static inline
int do_vfsub_mknod(struct inode *dir, struct dentry *dentry, int mode,
		   dev_t dev)
{
	LKTRTrace("i%lu, %.*s, 0x%x\n", dir->i_ino, DLNPair(dentry), mode);
	return vfs_mknod(dir, dentry, mode, dev);
}

static inline
int do_vfsub_link(struct dentry *src_dentry, struct inode *dir,
		  struct dentry *dentry)
{
	int err;

	LKTRTrace("%.*s, i%lu, %.*s\n",
		  DLNPair(src_dentry), dir->i_ino, DLNPair(dentry));

	lockdep_off();
	err = vfs_link(src_dentry, dir, dentry);
	lockdep_on();
	return err;
}

static inline
int do_vfsub_rename(struct inode *src_dir, struct dentry *src_dentry,
		    struct inode *dir, struct dentry *dentry)
{
	int err;

	LKTRTrace("i%lu, %.*s, i%lu, %.*s\n",
		  src_dir->i_ino, DLNPair(src_dentry),
		  dir->i_ino, DLNPair(dentry));

	lockdep_off();
	err = vfs_rename(src_dir, src_dentry, dir, dentry);
	lockdep_on();
	return err;
}

static inline
int do_vfsub_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	LKTRTrace("i%lu, %.*s, 0x%x\n", dir->i_ino, DLNPair(dentry), mode);
	return vfs_mkdir(dir, dentry, mode);
}

static inline int do_vfsub_rmdir(struct inode *dir, struct dentry *dentry)
{
	int err;

	LKTRTrace("i%lu, %.*s\n", dir->i_ino, DLNPair(dentry));

	lockdep_off();
	err = vfs_rmdir(dir, dentry);
	lockdep_on();
	return err;
}

static inline int do_vfsub_unlink(struct inode *dir, struct dentry *dentry)
{
	int err;

	LKTRTrace("i%lu, %.*s\n", dir->i_ino, DLNPair(dentry));

	lockdep_off();
	err = vfs_unlink(dir, dentry);
	lockdep_on();
	return err;
}

/* ---------------------------------------------------------------------- */

static inline
ssize_t do_vfsub_read_u(struct file *file, char __user *ubuf, size_t count,
			loff_t *ppos)
{
	ssize_t err;

	LKTRTrace("%.*s, cnt %lu, pos %Ld\n",
		  DLNPair(file->f_dentry), (unsigned long)count, *ppos);

	/* nfs uses some locks */
	lockdep_off();
	err = vfs_read(file, ubuf, count, ppos);
	lockdep_on();
	return err;
}

// kernel_read() ??
static inline
ssize_t do_vfsub_read_k(struct file *file, void *kbuf, size_t count,
			loff_t *ppos)
{
	ssize_t err;
	mm_segment_t oldfs;

	oldfs = get_fs();
	set_fs(KERNEL_DS);
	err = do_vfsub_read_u(file, (char __user*)kbuf, count, ppos);
	set_fs(oldfs);
	return err;
}

static inline
ssize_t do_vfsub_write_u(struct file *file, const char __user *ubuf,
			 size_t count, loff_t *ppos)
{
	ssize_t err;

	LKTRTrace("%.*s, cnt %lu, pos %Ld\n",
		  DLNPair(file->f_dentry), (unsigned long)count, *ppos);

	lockdep_off();
	err = vfs_write(file, ubuf, count, ppos);
	lockdep_on();
	return err;
}

static inline
ssize_t do_vfsub_write_k(struct file *file, void *kbuf, size_t count,
			 loff_t *ppos)
{
	ssize_t err;
	mm_segment_t oldfs;

	oldfs = get_fs();
	set_fs(KERNEL_DS);
	err = do_vfsub_write_u(file, (const char __user*)kbuf, count, ppos);
	set_fs(oldfs);
	return err;
}

static inline
int do_vfsub_readdir(struct file *file, filldir_t filldir, void *arg)
{
	int err;

	LKTRTrace("%.*s\n", DLNPair(file->f_dentry));

	lockdep_off();
	err = vfs_readdir(file, filldir, arg);
	lockdep_on();
	return err;
}

/* ---------------------------------------------------------------------- */

//#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 17)
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 23) && CONFIG_AUFS_SPLICE_PATCH
static inline
ssize_t do_vfsub_splice_read(struct file *file, loff_t *ppos,
			     struct pipe_inode_info *pipe, size_t len,
			     unsigned int flags)
{
	ssize_t err;

	LKTRTrace("%.*s, pos %Ld, len %Lu, 0x%x\n",
		  DLNPair(file->f_dentry), *ppos, len, flags);

#define vfs_splice_read	do_splice_to
	err = vfs_splice_read(file, ppos, pipe, len, flags);
	return err;
}

#if 0
int vfsub_splice_write()
{
}
#endif
#else
static inline
ssize_t do_vfsub_splice_read(struct file *file, loff_t *ppos,
			     struct pipe_inode_info *pipe, size_t len,
			     unsigned int flags)
{
	return -ENOSYS;
}
#endif

/* ---------------------------------------------------------------------- */

static inline loff_t vfsub_llseek(struct file *file, loff_t offset, int origin)
{
	loff_t err;

	LKTRTrace("%.*s\n", DLNPair(file->f_dentry));

	lockdep_off();
	err = vfs_llseek(file, offset, origin);
	lockdep_on();
	return err;
}

static inline int do_vfsub_getattr(struct vfsmount *mnt, struct dentry *dentry,
				   struct kstat *st)
{
	LKTRTrace("%.*s\n", DLNPair(dentry));
	return vfs_getattr(mnt, dentry, st);
}

/* ---------------------------------------------------------------------- */

#ifdef CONFIG_AUFS_DLGT
int vfsub_permission(struct inode *inode, int mask, struct nameidata *nd,
		     int dlgt);

int vfsub_create(struct inode *dir, struct dentry *dentry, int mode,
		 struct nameidata *nd, int dlgt);
int vfsub_symlink(struct inode *dir, struct dentry *dentry, const char *symname,
		  int mode, int dlgt);
int vfsub_mknod(struct inode *dir, struct dentry *dentry, int mode, dev_t dev,
		int dlgt);
int vfsub_link(struct dentry *src_dentry, struct inode *dir,
	       struct dentry *dentry, int dlgt);
int vfsub_rename(struct inode *src_dir, struct dentry *src_dentry,
		 struct inode *dir, struct dentry *dentry, int dlgt);
int vfsub_mkdir(struct inode *dir, struct dentry *dentry, int mode, int dlgt);
int vfsub_rmdir(struct inode *dir, struct dentry *dentry, int dlgt);

ssize_t vfsub_read_u(struct file *file, char __user *ubuf, size_t count,
		     loff_t *ppos, int dlgt);
ssize_t vfsub_read_k(struct file *file, void *kbuf, size_t count, loff_t *ppos,
		     int dlgt);
ssize_t vfsub_write_u(struct file *file, const char __user *ubuf, size_t count,
		      loff_t *ppos, int dlgt);
ssize_t vfsub_write_k(struct file *file, void *kbuf, size_t count, loff_t *ppos,
		      int dlgt);
int vfsub_readdir(struct file *file, filldir_t filldir, void *arg, int dlgt);
ssize_t vfsub_splice_read(struct file *file, loff_t *ppos,
			  struct pipe_inode_info *pipe, size_t len,
			  unsigned int flags, int dlgt);

int vfsub_getattr(struct vfsmount *mnt, struct dentry *dentry, struct kstat *st,
		  int dlgt);
#else

static inline
int vfsub_permission(struct inode *inode, int mask, struct nameidata *nd,
		     int dlgt)
{
	return do_vfsub_permission(inode, mask, nd);
}

static inline
int vfsub_create(struct inode *dir, struct dentry *dentry, int mode,
		 struct nameidata *nd, int dlgt)
{
	return do_vfsub_create(dir, dentry, mode, nd);
}

static inline
int vfsub_symlink(struct inode *dir, struct dentry *dentry, const char *symname,
		  int mode, int dlgt)
{
	return do_vfsub_symlink(dir, dentry, symname, mode);
}

static inline
int vfsub_mknod(struct inode *dir, struct dentry *dentry, int mode, dev_t dev,
		int dlgt)
{
	return do_vfsub_mknod(dir, dentry, mode, dev);
}

static inline
int vfsub_link(struct dentry *src_dentry, struct inode *dir,
	       struct dentry *dentry, int dlgt)
{
	return do_vfsub_link(src_dentry, dir, dentry);
}

static inline
int vfsub_rename(struct inode *src_dir, struct dentry *src_dentry,
		 struct inode *dir, struct dentry *dentry, int dlgt)
{
	return do_vfsub_rename(src_dir, src_dentry, dir, dentry);
}

static inline
int vfsub_mkdir(struct inode *dir, struct dentry *dentry, int mode,
		int dlgt)
{
	return do_vfsub_mkdir(dir, dentry, mode);
}

static inline
int vfsub_rmdir(struct inode *dir, struct dentry *dentry, int dlgt)
{
	return do_vfsub_rmdir(dir, dentry);
}

static inline
ssize_t vfsub_read_u(struct file *file, char __user *ubuf, size_t count,
		     loff_t *ppos, int dlgt)
{
	return do_vfsub_read_u(file, ubuf, count, ppos);
}

static inline
ssize_t vfsub_read_k(struct file *file, void *kbuf, size_t count, loff_t *ppos,
		     int dlgt)
{
	return do_vfsub_read_k(file, kbuf, count, ppos);
}

static inline
ssize_t vfsub_write_u(struct file *file, const char __user *ubuf, size_t count,
		      loff_t *ppos, int dlgt)
{
	return do_vfsub_write_u(file, ubuf, count, ppos);
}

static inline
ssize_t vfsub_write_k(struct file *file, void *kbuf, size_t count, loff_t *ppos,
		      int dlgt)
{
	return do_vfsub_write_k(file, kbuf, count, ppos);
}

static inline
int vfsub_readdir(struct file *file, filldir_t filldir, void *arg, int dlgt)
{
	return do_vfsub_readdir(file, filldir, arg);
}

static inline
ssize_t vfsub_splice_read(struct file *file, loff_t *ppos,
			  struct pipe_inode_info *pipe, size_t len,
			  unsigned int flags, int dlgt)
{
	return do_vfsub_splice_read(file, ppos, pipe, len, flags);
}

static inline int vfsub_getattr(struct vfsmount *mnt, struct dentry *dentry,
				struct kstat *st, int dlgt)
{
	return do_vfsub_getattr(mnt, dentry, st);
}
#endif /* CONFIG_AUFS_DLGT */

/* ---------------------------------------------------------------------- */

static inline
struct dentry *vfsub_lock_rename(struct dentry *d1, struct dentry *d2)
{
	struct dentry *d;

	lockdep_off();
	d = lock_rename(d1, d2);
	lockdep_on();
	return d;
}

static inline void vfsub_unlock_rename(struct dentry *d1, struct dentry *d2)
{
	lockdep_off();
	unlock_rename(d1, d2);
	lockdep_on();
}

/* ---------------------------------------------------------------------- */

int vfsub_notify_change(struct dentry *dentry, struct iattr *ia, int dlgt);
int vfsub_unlink(struct inode *dir, struct dentry *dentry, int dlgt);
int vfsub_statfs(void *arg, struct kstatfs *buf, int dlgt);

#endif /* __KERNEL__ */
#endif /* __AUFS_VFSUB_H__ */
